package com.xebialabs.xlrelease.notifications.email;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import jakarta.activation.DataHandler;
import jakarta.activation.DataSource;
import jakarta.mail.*;
import jakarta.mail.internet.*;
import jakarta.mail.util.ByteArrayDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.configuration.CustomLogoSettings;
import com.xebialabs.xlrelease.domain.notification.MailPriority;
import com.xebialabs.xlrelease.notifications.configuration.SmtpServer;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.service.ConfigurationService;
import com.xebialabs.xlrelease.spring.configuration.ExecutorNames;

import static com.xebialabs.xlrelease.domain.notification.MailPriority.Normal;
import static jakarta.mail.internet.InternetAddress.parse;
import static java.util.Arrays.asList;

@Component
public class EmailSender {
    private static final Logger logger = LoggerFactory.getLogger(EmailSender.class);

    private static final String LOGO_PATH = "/email-logo-digital-ai-release.png";

    private final XlrConfig xlrConfig;
    private final ConfigurationRepository configurationRepository;
    private final EmailFactory emailFactory;
    private final ConfigurationService configurationService;
    private final ExecutorService auxiliaryExecutor;

    public EmailSender(ConfigurationRepository configurationRepository,
                       EmailFactory emailFactory,
                       XlrConfig xlrConfig,
                       ConfigurationService configurationService,
                       @Qualifier(ExecutorNames.AUXILIARY_EXECUTOR_NAME)
                       ExecutorService auxiliaryExecutor
    ) {
        this.configurationRepository = configurationRepository;
        this.emailFactory = emailFactory;
        this.xlrConfig = xlrConfig;
        this.configurationService = configurationService;
        this.auxiliaryExecutor = auxiliaryExecutor;
    }

    // used in remote completion plugin
    public CompletableFuture<Void> scheduleEmail(final Email email) {
        CompletableFuture<Void> future = new CompletableFuture<>();
        auxiliaryExecutor.submit(() -> {
            try {
                sendEmailSync(email);
                future.complete(null);
            } catch (Exception e) {
                future.completeExceptionally(e);
            }
        });
        return future;
    }

    public void sendEmailSync(final Email email) throws MessagingException, NotFoundException {
        if (xlrConfig.maintenanceModeEnabled()) {
            logger.warn("[MAINTENANCE MODE] Email notification '{}' - DROPPED", email.subject);
            return;
        }
        if (email.getRecipients() == null || email.getRecipients().isEmpty()) {
            String msg = String.format("Email with subject '%s' not sent, because the recipient is missing.", email.getSubject());
            logger.warn(msg);
            throw new IllegalArgumentException(msg);
        } else {
            try {
                sendHTML(email);
            } catch (NotFoundException notFoundException) {
                logger.info("Could not send email: {}. Please configure the SMTP Server", notFoundException.toString());
                throw notFoundException;
            } catch (MessagingException e) {
                logger.error("Could not send email ", e);
                throw e;
            }
        }
    }

    void sendHTML(Email email) throws MessagingException {
        SmtpServer smtpServer = configurationRepository.read(SmtpServer.SMTP_SERVER_ID());
        sendHTML(smtpServer, email);
    }

    public void sendHTML(SmtpServer smtpServer, Email email) throws MessagingException {
        MimeMessage mimeEmail = emailFactory.newMimeEmail(smtpServer, email.getSenderUsername(), email.getSenderPassword());

        setRecipients(mimeEmail, email.getRecipients());

        if (email.getFrom().isPresent() && !email.getFrom().get().isEmpty()) {
            setFrom(mimeEmail, email.getFrom().orElse(""));
        } else {
            setFrom(mimeEmail, smtpServer.getFromAddress());
        }

        setMailPriority(mimeEmail, email.getPriority().orElse(Normal));

        if (email.getBcc().isPresent() && !email.getBcc().get().isEmpty()) {
            mimeEmail.addRecipients(Message.RecipientType.BCC, toInternetAddresses(email.getBcc().orElse(Collections.emptyList())));
        }

        if (email.getCc().isPresent() && !email.getCc().get().isEmpty()) {
            mimeEmail.addRecipients(Message.RecipientType.CC, toInternetAddresses(email.getCc().orElse(Collections.emptyList())));
        }

        if (email.getReplyTo().isPresent() && !email.getReplyTo().get().isEmpty()) {
            InternetAddress[] parse = parse(email.getReplyTo().orElse(""));
            mimeEmail.setReplyTo(parse);
        }

        mimeEmail.setSubject(email.getSubject());

        Multipart multipart = new MimeMultipart();

        MimeBodyPart emailBodyPart = new MimeBodyPart();
        emailBodyPart.setContent(email.getBody(), "text/html; charset=UTF-8");
        multipart.addBodyPart(emailBodyPart);

        MimeBodyPart logoPart = new MimeBodyPart();
        logoPart.setContentID("logo");
        logoPart.setDisposition(Part.INLINE);
        logoPart.setDataHandler(new DataHandler(getClass().getResource(LOGO_PATH)));
        multipart.addBodyPart(logoPart);

        try {
            MimeBodyPart customLogo = new MimeBodyPart();
            CustomLogoSettings customLogoSettings = this.readCustomLogo();
            byte[] customLogoByteArray = Base64.getDecoder().decode(customLogoSettings.getContent());
            DataSource customLogoDataSource = new ByteArrayDataSource(customLogoByteArray, customLogoSettings.getContentType());
            customLogo.setContentID("customLogo");
            customLogo.setDisposition(Part.INLINE);
            customLogo.setDataHandler(new DataHandler(customLogoDataSource));
            multipart.addBodyPart(customLogo);
        } catch (NotFoundException e) {
            logger.debug("Custom logo is not found");
        }

        mimeEmail.setContent(multipart);
        Transport.send(mimeEmail);
    }

    private CustomLogoSettings readCustomLogo() {
        CustomLogoSettings customLogoSettings = (CustomLogoSettings) configurationService.read(CustomLogoSettings.CUSTOM_LOGO_SETTINGS_ID);
        if (customLogoSettings.getContent() == null) {
            throw new NotFoundException("Custom logo is not found");
        }
        return customLogoSettings;
    }

    void setMailPriority(MimeMessage mimeEmail, MailPriority mailPriority) throws MessagingException {
        mimeEmail.addHeader("X-Priority", mailPriority.getPriority());
        mimeEmail.addHeader("Importance", mailPriority.getImportance());
    }

    @VisibleForTesting
    void setRecipients(MimeMessage mimeEmail, Collection<String> recipients) throws MessagingException {
        mimeEmail.setRecipients(Message.RecipientType.TO, toInternetAddresses(recipients));
    }

    @VisibleForTesting
    InternetAddress[] toInternetAddresses(Collection<String> recipients) throws IllegalArgumentException {
        List<InternetAddress> addresses = new ArrayList<>();
        for (String recipient : recipients) {
            try {
                addresses.addAll(asList(parse(recipient)));
            } catch (AddressException e) {
                throw new IllegalArgumentException(e);
            }
        }

        return addresses.toArray(new InternetAddress[0]);
    }

    @VisibleForTesting
    void setFrom(MimeMessage mimeEmail, String fromAddress) throws IllegalArgumentException {
        if (fromAddress == null) {
            return;
        }

        try {
            InternetAddress from = new InternetAddress(fromAddress);
            mimeEmail.setFrom(from);
        } catch (MessagingException e) {
            throw new IllegalArgumentException(e);
        }
    }
}
