package com.xebialabs.xlrelease.notifications.handlers

import com.xebialabs.xlrelease.domain.events.{CommentCreatedEvent, CommentUpdatedEvent}
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.notifications.mentions.Mentions.collectMentions
import com.xebialabs.xlrelease.notifications.{CommentAdded, NotificationService, UserMentioned}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@EventListener
class CommentNotificationEventHandler @Autowired()(notificationService: NotificationService) {

  @AsyncSubscribe
  def onCommentCreated(event: CommentCreatedEvent): Unit = {
    val (task, comment, createdByUser) = (event.task, event.comment, event.createdByUser)

    if (task.getRelease.isActive && createdByUser) {
      notificationService.notify(CommentAdded(task, comment))
    }

    val mentions = collectMentions(comment.getText)

    if (mentions.nonEmpty) {
      notificationService.notify(UserMentioned(task, mentions, event.username, comment.getText))
    }
  }

  @AsyncSubscribe
  def onCommentUpdated(event: CommentUpdatedEvent): Unit = {
    val (original, updated) = (event.original, event.updated)
    val originalMentions = collectMentions(original.getText)
    val updatedMentions = collectMentions(updated.getText)

    val newMentions = updatedMentions -- originalMentions
    if (newMentions.nonEmpty) {
      notificationService.notify(UserMentioned(event.task, newMentions, event.username, updated.getText))
    }
  }

}
