package com.xebialabs.xlrelease.notifications.api.internal

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.notifications.TriggerType
import com.xebialabs.xlrelease.notifications.configuration.EmailNotificationSettings.GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID
import com.xebialabs.xlrelease.notifications.configuration.trigger.NotificationTriggerSettings
import com.xebialabs.xlrelease.notifications.email.EmailNotificationsService
import com.xebialabs.xlrelease.notifications.views.converters.EmailNotificationSettingsConverter.{fromView, toView}
import com.xebialabs.xlrelease.notifications.views.{EmailNotificationSettingsView, EmailPreviewView}
import com.xebialabs.xlrelease.repository.Ids.{ROOT_FOLDER_ID, isRoot}
import com.xebialabs.xlrelease.security.PermissionChecker
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller
import org.springframework.util.StringUtils

import jakarta.ws.rs._
import jakarta.ws.rs.core.MediaType.APPLICATION_JSON

@Path("/settings/notifications")
@Produces(Array(APPLICATION_JSON))
@Controller
class NotificationsResource @Autowired()(emailNotificationsService: EmailNotificationsService,
                                         permissions: PermissionChecker,
                                         serverConfiguration: ServerConfiguration) {

  private val serverUrl = serverConfiguration.getServerUrl

  @GET
  @Path("emails")
  def getEmailNotificationSettings: EmailNotificationSettingsView = {
    permissions.check(ADMIN)
    toView(emailNotificationsService.getSettings(GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID))
  }

  @POST
  @Path("emails")
  def updateEmailNotificationSettings(emailNotificationSettingsView: EmailNotificationSettingsView): EmailNotificationSettingsView = {
    permissions.check(ADMIN)

    emailNotificationSettingsView.setId(GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID)
    emailNotificationSettingsView.setFolderId(ROOT_FOLDER_ID)

    toView(emailNotificationsService.updateSettings(fromView(emailNotificationSettingsView)))
  }

  @POST
  @Path("emails/triggerSettings")
  def updateEmailTemplateSettings(notificationTriggerSettings: NotificationTriggerSettings): Unit = {
    permissions.check(ADMIN)
    val settings = emailNotificationsService.getSettings(GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID)
    settings.setNotificationTriggerSettings(notificationTriggerSettings)
    emailNotificationsService.updateSettings(settings)
  }

  @POST
  @Path("emails/preview")
  def preview(@QueryParam("folderId") folderId: String, view: EmailPreviewView): String = {
    if (!StringUtils.isEmpty(folderId) && !isRoot(folderId)) {
      permissions.checkEditNotification(folderId)
    } else {
      permissions.check(ADMIN)
    }

    emailNotificationsService.createPreview(TriggerType.withName(view.triggerType), view.body, serverUrl)
  }

  @GET
  @Timed
  @Path("emails/{settingsId:.*/EmailNotificationSettings[^/]*}")
  def getEmailNotificationSettings(@PathParam("settingsId") settingsId: String): EmailNotificationSettingsView = {
    val settings = emailNotificationsService.getSettings(settingsId)
    permissions.checkEditNotification(settings.getFolderId)
    toView(settings)
  }

  @GET
  @Timed
  @Path("emails/{folderId:.*/Folder[^/]*}")
  def getEmailNotificationSettingsForFolder(@PathParam("folderId") folderId: String): EmailNotificationSettingsView = {
    permissions.checkEditNotification(folderId)
    toView(emailNotificationsService.getSettings(folderId), emailNotificationsService.getInheritableFolderId(folderId))
  }

  @POST
  @Timed
  @Path("emails/{folderId:.*/Folder[^/]*}")
  def addEmailNotificationSettings(@PathParam("folderId") folderId: String,
                                   emailNotificationSettingsView: EmailNotificationSettingsView): EmailNotificationSettingsView = {
    permissions.checkEditNotification(folderId)

    emailNotificationSettingsView.setFolderId(folderId)
    toView(emailNotificationsService.create(fromView(emailNotificationSettingsView)))
  }

  @PUT
  @Timed
  @Path("emails/{settingsId:.*/EmailNotificationSettings[^/]*}")
  def updateEmailNotificationSettings(@PathParam("settingsId") settingsId: String,
                                      emailNotificationSettingsView: EmailNotificationSettingsView): EmailNotificationSettingsView = {
    val settings = emailNotificationsService.getSettings(settingsId)

    permissions.checkEditNotification(settings.getFolderId)

    emailNotificationSettingsView.setId(settingsId)
    toView(emailNotificationsService.updateSettings(fromView(emailNotificationSettingsView)))
  }

  @PUT
  @Timed
  @Path("emails/{settingsId:.*/EmailNotificationSettings[^/]*}/triggerSettings")
  def updateEmailTemplateSettings(@PathParam("settingsId") settingsId: String,
                                  notificationTriggerSettings: NotificationTriggerSettings): Unit = {
    val settings = emailNotificationsService.getSettings(settingsId)

    permissions.checkEditNotification(settings.getFolderId)

    settings.setNotificationTriggerSettings(notificationTriggerSettings)
    emailNotificationsService.updateSettings(settings)
  }

  @DELETE
  @Timed
  @Path("emails/{settingsId:.*/EmailNotificationSettings[^/]*}")
  def deleteEmailNotificationSettings(@PathParam("settingsId") settingsId: String): Unit = {
    val settings = emailNotificationsService.getSettings(settingsId)

    if (settings.getId == GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID) {
      throw new IllegalArgumentException("Can't delete the global email notification settings")
    }
    permissions.checkEditNotification(settings.getFolderId)

    emailNotificationsService.deleteSettings(settings)
  }

}
