package com.xebialabs.xlrelease.notifications.initialize

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.server.api.upgrade.RepositoryInitialization
import com.xebialabs.xlrelease.domain.notification.MailPriority
import com.xebialabs.xlrelease.notifications.TriggerType.{RELEASE_ABORTED, RELEASE_COMPLETED, RELEASE_FLAGGED, RELEASE_STARTED, _}
import com.xebialabs.xlrelease.notifications.configuration.EmailNotificationSettings
import com.xebialabs.xlrelease.notifications.configuration.EmailNotificationSettings.GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID
import com.xebialabs.xlrelease.notifications.configuration.trigger.RecipientSettings._
import com.xebialabs.xlrelease.notifications.configuration.trigger.{NotificationTriggerSettings, RecipientSettings, TemplateSettings}
import com.xebialabs.xlrelease.notifications.initialize.EmailNotificationSettingsInitializer.allNotificationTriggerSettings
import com.xebialabs.xlrelease.service.ConfigurationService
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.common.ClassNameToString
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.Collections.emptyList
import scala.jdk.CollectionConverters._

@Component
class EmailNotificationSettingsInitializer @Autowired()(configurationService: ConfigurationService)
  extends RepositoryInitialization with Logging with ClassNameToString {

  override def getComponent: String = XL_RELEASE_COMPONENT

  override def doInitialize(): Unit = {
    val settings: EmailNotificationSettings = Type.valueOf(classOf[EmailNotificationSettings]).getDescriptor.newInstance(GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID)
    settings.setNotificationTriggerSettings(allNotificationTriggerSettings.asJava)

    logger.info("Creating default email notification settings")

    configurationService.createOrUpdate(settings)

    logger.debug("Finished creating default email notification settings")
  }
}

object EmailNotificationSettingsInitializer {
  final val PRODUCT_TAG = "[Release]"

  private def allNotificationTriggerSettings: List[NotificationTriggerSettings] = {
    releaseNotificationTriggerSettings ++ taskNotificationTriggerSettings
  }

  private def releaseNotificationTriggerSettings() = {
    val ReleaseTitle = " ${release.title}"
    List(
      notificationTriggerSettings(
        RELEASE_STARTED,
        roles = List(RELEASE_ADMIN_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTitle,
        body = "The **${release.title}** release has been started."),
      notificationTriggerSettings(
        RELEASE_ABORTED,
        roles = List(RELEASE_ADMIN_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTitle,
        body = "The **${release.title}** release has been aborted."),
      notificationTriggerSettings(
        RELEASE_COMPLETED,
        roles = List(RELEASE_ADMIN_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTitle,
        body = "The **${release.title}** release has been completed."),
      notificationTriggerSettings(
        RELEASE_FLAGGED,
        roles = List(RELEASE_ADMIN_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTitle,
        body =
          """
            |The status of the **${release.title}** release was changed to **${release.flagStatus}** with the following comment:
            |
            |**${release.flagComment}**
          """.stripMargin.trim,
        priority = MailPriority.High),
      notificationTriggerSettings(
        RELEASE_FAILED,
        roles = List(RELEASE_ADMIN_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTitle,
        body =
          """
            |The **${release.title}** release has failed due to a task failure.
            |
            |The release is now stopped.
            |
            |You can retry the failed task, reassign it to somebody else or abort the release.
          """.stripMargin.trim,
        priority = MailPriority.High),
      notificationTriggerSettings(
        RELEASE_FAILING,
        roles = List(RELEASE_ADMIN_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTitle,
        body =
          """
            |The **${release.title}** release is failing due to a task failure.
            |
            |Other tasks may still be in progress, but the release will stop after they complete.
            |
            |You can retry the failed task, reassign it to somebody else or abort the release.
          """.stripMargin.trim,
        priority = MailPriority.High)
    )
  }

  //noinspection ScalaStyle
  private def taskNotificationTriggerSettings = {
    val ReleaseTaskTitle = " ${release.title}: ${task.title}"
    List(
      notificationTriggerSettings(
        ACTIVE_TASK_ASSIGNED,
        roles = List(TASK_OWNER_ROLE_NAME, TASK_TEAM_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The active task **${task.title}** in the **${release.title}** release is now assigned to ${task.ownerFullName}.
          """.stripMargin.trim),
      notificationTriggerSettings(
        MANUAL_TASK_STARTED,
        roles = List(TASK_OWNER_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The **${task.title}** task in the **${release.title}** release has started and is assigned to you or your team.
            |
            |After the task is completed, please mark it as Completed in Release.
          """.stripMargin.trim),
      notificationTriggerSettings(
        TASK_FAILED,
        roles = List(TASK_OWNER_ROLE_NAME, TASK_TEAM_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The **${task.title}** task in the **${release.title}** release has failed.
            |
            |Other tasks could still be in progress, but the release will stop after they complete.
            |
            |You can retry the task, reassign it to somebody else or abort the release.
          """.stripMargin.trim,
        priority = MailPriority.High),
      notificationTriggerSettings(
        COMMENT_ADDED,
        roles = List(TASK_OWNER_ROLE_NAME, TASK_TEAM_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |**${comment.authorFullName}** commented on the **${task.title}** task in the **${release.title}** release:
            |
            |${comment.text}
          """.stripMargin.trim),
      notificationTriggerSettings(
        TASK_WAITING_FOR_INPUT,
        roles = List(TASK_OWNER_ROLE_NAME, TASK_TEAM_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The **${task.title}** task in the **${release.title}** release needs your input.
            |
            |Please enter the required information so the release can continue.
          """.stripMargin.trim),
      notificationTriggerSettings(
        MANUAL_TASK_STARTED_WITHOUT_OWNER,
        roles = List(RELEASE_ADMIN_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The **${task.title}** task in the **${release.title}** release has started, but it is not assigned.
            |
            |Please assign the task to a user or a team.
          """.stripMargin.trim),
      notificationTriggerSettings(
        ACTIVE_TASK_UNASSIGNED,
        roles = List(TASK_OWNER_ROLE_NAME, TASK_TEAM_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The **${task.title}** task in the **${release.title}** release is active but not assigned to anyone anymore.
            |
            |Please assign the task to a user or a team.
          """.stripMargin.trim),
      notificationTriggerSettings(
        TASK_OVERDUE,
        roles = List(TASK_OWNER_ROLE_NAME, RELEASE_ADMIN_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The **${task.title}** task in the **${release.title}** release is overdue.
            |
            |After the task is completed, please mark it as Completed in Release.
          """.stripMargin.trim,
        priority = MailPriority.High),
      notificationTriggerSettings(
        TASK_FLAGGED,
        roles = List(TASK_TEAM_ROLE_NAME, RELEASE_ADMIN_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The status of the **${task.title}** task in the **${release.title}** release was changed to **${task.flagStatus}** with the following comment:
            |
            |**${release.flagComment}**
          """.stripMargin.trim,
        priority = MailPriority.High),
      notificationTriggerSettings(
        TASK_DUE_SOON,
        roles = List(TASK_OWNER_ROLE_NAME, RELEASE_ADMIN_ROLE_NAME, WATCHER_ROLE_NAME),
        subject = PRODUCT_TAG + ReleaseTaskTitle,
        body =
          """
            |The **${task.title}** task in the **${release.title}** release is due in **${task.dueInHours}** hours and **${task.dueInMinutes}** minutes.
            |
            |After the task is completed, please mark it as Completed in Release.
          """.stripMargin.trim
      )
    )
  }

  def notificationTriggerSettings(notificationTriggerName: String,
                                  roles: List[String] = List.empty,
                                  subject: String,
                                  body: String,
                                  priority: MailPriority = MailPriority.Normal): NotificationTriggerSettings = {
    val settings = new NotificationTriggerSettings
    val recipientSettings = new RecipientSettings
    val templateSettings = new TemplateSettings

    settings.setTemplateSettings(templateSettings)
    settings.setRecipientSettings(recipientSettings)

    settings.setNotificationTriggerName(notificationTriggerName)
    settings.setPriority(priority)

    recipientSettings.setRoles(roles.asJava)
    recipientSettings.setGlobalRoles(emptyList())
    recipientSettings.setUsers(emptyList())

    templateSettings.setSubject(subject)
    templateSettings.setBody(body)

    settings
  }
}
