package com.xebialabs.xlrelease.notifications.email

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.{PlanItem, Release, Task}
import com.xebialabs.xlrelease.notifications.TriggerType.{TriggerType, _}
import com.xebialabs.xlrelease.notifications.configuration.EmailNotificationSettings
import com.xebialabs.xlrelease.notifications.configuration.EmailNotificationSettings._
import com.xebialabs.xlrelease.notifications.configuration.trigger.{NotificationTriggerSettings, RecipientSettings}
import com.xebialabs.xlrelease.notifications.email.templates.Templater
import com.xebialabs.xlrelease.notifications.{ReleaseNotificationTrigger, _}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{MAIL_CONFIGURATION_ROOT, findFolderId}
import com.xebialabs.xlrelease.service.{CiIdService, ConfigurationService}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class EmailNotificationsService @Autowired()(ciIdService: CiIdService,
                                             emailNotificationCache: EmailNotificationCache,
                                             configurationService: ConfigurationService,
                                             contextHelper: ContextHelper,
                                             emailRecipientResolver: EmailRecipientResolver,
                                             emailSender: EmailSender) extends NotificationStrategy with Logging {
  def getInheritableFolderId(folderId: String): String = {
    if (Ids.isFolderId(Ids.getParentId(folderId))) {
      emailNotificationCache.getSettings(Ids.getParentId(folderId)).getFolderId
    } else {
      null
    }
  }

  def getSettings(containerId: String): EmailNotificationSettings = {
    emailNotificationCache.getSettings(containerId)
  }

  val EXCLUDE_AUTHOR_TRIGGERS: Set[TriggerType] = Set(COMMENT_ADDED, RELEASE_FLAGGED, TASK_FLAGGED)

  /**
   * Clear complete cache on any create, update or delete.
   * Method will be called via actor to make clear cache on all the clustered nodes
   */
  def clearCache(): Unit = {
    emailNotificationCache.clearCache()
  }

  def createPreview(triggerType: TriggerType, body: String, serverUrl: String = "${url}"): String = {
    Templater.createPreviewForTriggerType(triggerType, contextHelper.baseContext, body, serverUrl)
  }


  def create(settings: EmailNotificationSettings): EmailNotificationSettings = {
    val id = ciIdService.getUniqueId(Type.valueOf(classOf[EmailNotificationSettings]), MAIL_CONFIGURATION_ROOT)
    settings.setId(id)
    configurationService.createOrUpdate(settings)
    configurationService.read(id).asInstanceOf[EmailNotificationSettings]
  }


  def updateSettings(settings: EmailNotificationSettings): EmailNotificationSettings = {
    if (settings.getId() == GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID) {
      validateStandaloneSettings(settings)
    }
    configurationService.createOrUpdate(settings)
    configurationService.read(settings.getId).asInstanceOf[EmailNotificationSettings]
  }

  def deleteSettings(settings: EmailNotificationSettings): Unit = {
    configurationService.delete(settings.getId)
  }

  // scalastyle:off method.length
  def notify(notificationTrigger: NotificationTrigger): Unit = {
    val triggerType = notificationTrigger.name

    def sendMail[A](recipientsToItems: Map[Recipients, List[A]],
                    contextGetter: List[A] => JavaMustacheContext,
                    triggerSettings: NotificationTriggerSettings,
                    sendIndependently: Boolean = false): Unit = {
      for ((recipients, items) <- recipientsToItems) {
        val context = contextGetter(items)

        val (subject, body) = items match {
          case _ :: Nil =>
            (
              Templater.processString(triggerSettings.getTemplateSettings.getSubject, context),
              Templater.processTemplateForTriggerType(triggerType, context, triggerSettings.getTemplateSettings.getBody)
            )
          case _ =>
            (
              Templater.processString(triggerSettings.getBulkTemplateSettings.getSubject, context),
              Templater.processBulkTemplateForTriggerType(triggerType, context, triggerSettings.getBulkTemplateSettings.getBody)
            )
        }

        val emails = if (sendIndependently) {
          recipients.map(recipient => new Email(Seq(recipient).asJava, subject, body).withPriority(triggerSettings.getPriority))
        } else {
          Seq(new Email(recipients.asJava, subject, body).withPriority(triggerSettings.getPriority))
        }
        emails.foreach(mail => emailSender.sendEmailSync(mail))
      }
    }

    val resolver: (PlanItem, RecipientSettings) => Recipients = resolveRecipients(triggerType)

    def getRecipientsToItems[A <: PlanItem](items: List[A], recipientSettings: RecipientSettings): Map[Recipients, List[A]] =
      reverseItemToRecipients(items.map(item => item -> resolver(item, recipientSettings)))

    def getNotificationTriggerSettings(id: String = GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID) = {
      getSettings(id).getNotificationTriggerSettings(triggerType.toString)
    }

    notificationTrigger match {
      case t: ReleaseNotificationTrigger =>
        val settings: NotificationTriggerSettings = getNotificationTriggerSettings(findFolderId(t.releases.head.getId))
        sendMail(getRecipientsToItems(t.releases, settings.getRecipientSettings), contextHelper.releasesContext, settings)
      case t: TaskNotificationTrigger =>
        val settings: NotificationTriggerSettings = getNotificationTriggerSettings(findFolderId(t.tasks.head.getId))
        sendMail(getRecipientsToItems(t.tasks, settings.getRecipientSettings), contextHelper.tasksContext, settings)
      case CommentAdded(tasks, comment) =>
        val settings: NotificationTriggerSettings = getNotificationTriggerSettings(findFolderId(tasks.head.getId))
        sendMail(getRecipientsToItems(tasks, settings.getRecipientSettings), contextHelper.commentsContext(comment), settings)
      case t: ReportJobNotificationTrigger =>
        val settings: NotificationTriggerSettings = getNotificationTriggerSettings()
        sendMail(Map(Set(emailRecipientResolver.resolveRecipient(t.owner)) -> List(t)), contextHelper.reportContext, settings)
      case UserMentioned(tasks, mentions, mentionedBy, mentionText) =>
        val settings: NotificationTriggerSettings = getNotificationTriggerSettings()
        val recipients = resolveRecipients(triggerType, mentions.toList)(tasks.head, settings.getRecipientSettings)
        if (recipients.nonEmpty) {
          sendMail(Map(recipients -> List(tasks.head)), contextHelper.mentionContext(mentionedBy, mentionText), settings, sendIndependently = true)
        }
      case u: UserTokenAboutToExpire =>
        val settings: NotificationTriggerSettings = getNotificationTriggerSettings()
        sendMail(Map(Set(emailRecipientResolver.resolveRecipient(u.username)) -> List(u)), contextHelper.userTokenContext, settings)
      case u: GenericSystemNotification =>
        val settings: NotificationTriggerSettings = getNotificationTriggerSettings()
        val recipients: Set[String] = emailRecipientResolver.resolveSystemAdminRecipients()
        sendMail(Map(recipients -> List(u)), contextHelper.genericNotificationContext, settings)
      case _ => ()
    }
  }
  // scalastyle:on method.length

  /*
    Reverses the mapping of a list of one-to-many relationships.
    We use this on a list of (Item -> {Recipient}) mappings to find out which recipients receive which items in their e-mail
    Step 1 (initial mapping):
      [
        (i1 -> {r1, r2, r3}),
        (i2 -> {r1}),
        (i3 -> {r2, r3})
      ]
    Step 2 (reverse mappings):
      [
        (r1 -> [i1, i2]),
        (r2 -> [i1, i3]),
        (r3 -> [i1, i3])
      ]
    Step 3 (combine keys with equal values):
      [
        ({r1} -> [i1, i2]),
        ({r2, r3} -> [i1, i3])
      ]
     */
  private[email] def reverseItemToRecipients[A](itemToRecipientsSet: List[(A, Recipients)]): Map[Recipients, List[A]] =
    itemToRecipientsSet
      .flatMap { case (item, recipients) => recipients.map(_ -> item) }
      .groupBy(_._1).view.mapValues(_.map(_._2)) // Step 2
      .groupBy(_._2).view.mapValues(_.toMap.keys.toSet) // Step 3
      .map(_.swap)
      .toMap

  private[email] def resolveRecipients(triggerType: TriggerType, additionalUsers: List[String] = List.empty)
                                      (planItem: PlanItem, recipientSettings: RecipientSettings): Recipients = {
    if (isDisableNotifications(planItem)) {
      Set.empty
    } else {
      val (roles, globalRoles, users) = (
        recipientSettings.getRoles.asScala.toList,
        List.empty,
        additionalUsers
      )
      emailRecipientResolver.resolveRecipients(roles, globalRoles, users, planItem, EXCLUDE_AUTHOR_TRIGGERS.contains(triggerType))
    }
  }

  private def isDisableNotifications(planItem: PlanItem): Boolean = {
    planItem match {
      case task: Task =>
        task.getRelease.isDisableNotifications
      case release: Release =>
        release.isDisableNotifications
      case _ =>
        false
    }
  }

  /**
   * Checks if standalone settings are present or not
   *
   * @param settings
   */
  def validateStandaloneSettings(settings: EmailNotificationSettings): Unit = {
    val standaloneTriggerTypes: List[TriggerType] = List(AUDIT_REPORT_JOB_COMPLETED, AUDIT_REPORT_JOB_ABORTED, AUDIT_REPORT_JOB_FAILED, USER_MENTIONED)
    standaloneTriggerTypes.foreach { triggerType =>
      if (!settings.hasNotificationTriggerSettings(triggerType.toString)) {
        throw new IllegalArgumentException(s"Email notification settings required for [${triggerType.toString}]")
      }
    }
  }
}

