package com.xebialabs.xlrelease.notifications.workers

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.FailureReasons.NOTIFICATION_TASK_FAILED
import com.xebialabs.xlrelease.domain.NotificationTask
import com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED
import com.xebialabs.xlrelease.notifications.email.templates.Templater.processTemplate
import com.xebialabs.xlrelease.notifications.email.{ContextHelper, Email, EmailSender}
import com.xebialabs.xlrelease.scheduler.NotificationTaskJob
import com.xebialabs.xlrelease.scheduler.workers.Worker
import com.xebialabs.xlrelease.scheduler.workers.Worker._
import com.xebialabs.xlrelease.script.TaskSoftReference
import com.xebialabs.xlrelease.user.User.SYSTEM
import com.xebialabs.xlrelease.utils.MarkdownProcessor.process
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
class NotificationTaskWorker(releaseActorService: ReleaseActorService,
                             contextHelper: ContextHelper,
                             emailSender: EmailSender,
                            ) extends Worker with Logging {

  override def execute: ExecuteJob = {
    case NotificationTaskJob(taskRef) =>
      sendNotification(taskRef)
  }

  override def processResult: ProcessJobResult = {
    case result: NotificationTaskExecutionResult => result match {
      case NotificationTaskExecutionResulSuccess(taskId, _) =>
        releaseActorService.markTaskAsDone(COMPLETED, taskId, null, SYSTEM)
      case NotificationTaskExecutionResulFailure(taskId, _, message) =>
        releaseActorService.failTaskAsync(taskId, NOTIFICATION_TASK_FAILED.format(message), SYSTEM)
    }
  }

  private def sendNotification(taskRef: TaskSoftReference[NotificationTask]): NotificationTaskExecutionResult = {
    val task = taskRef.get()
    Try {
      val context = contextHelper.baseContext + ("body" -> process(task.getBody).orElse(null))
      val body = processTemplate("notification-task-template.mustache", context.asJava)
      val email = new Email(task.getAddresses, task.getSubject, body)

      email.withPriority(task.getPriority)
        .withReplyTo(task.getReplyTo)
        .withCc(task.getCc)
        .withBcc(task.getBcc)
        .withFrom(task.getFrom)
        .withSenderUsername(task.getSenderUsername)
        .withSenderPassword(task.getSenderPassword)

      Map(
        "To" -> task.getAddresses.isEmpty,
        "Subject" -> isNullOrEmpty(task.getSubject),
        "Body" -> isNullOrEmpty(task.getBody)).filter(_._2).keys.toList match {
        case Nil =>
          Try(emailSender.sendEmailSync(email)) match {
            case Success(_) => NotificationTaskExecutionResulSuccess(task.getId, task.getExecutionId)
            case Failure(e: Throwable) => NotificationTaskExecutionResulFailure(task.getId, task.getExecutionId, e.getMessage)
          }
        case missedFields =>
          val errorMessage = s"The required fields [${missedFields.mkString(", ")}] are missed"
          logger.warn(s"Email not sent for notification task '${task.getId}'. $errorMessage")
          NotificationTaskExecutionResulFailure(task.getId, task.getExecutionId, errorMessage)
      }
    }.recover { e =>
      NotificationTaskExecutionResulFailure(task.getId, task.getExecutionId, e.getMessage)
    }.get
  }

}
