package com.xebialabs.xlrelease.notifications.email

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.configuration.CustomLogoSettings
import com.xebialabs.xlrelease.domain.{Comment, Release, Task}
import com.xebialabs.xlrelease.notifications.mentions.Mentions.resolveMentions
import com.xebialabs.xlrelease.notifications.{GenericSystemNotification, ReportJobNotificationTrigger, UserTokenAboutToExpire}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.IdType.DOMAIN
import com.xebialabs.xlrelease.service.UserInfoResolver
import com.xebialabs.xlrelease.utils.MarkdownProcessor
import org.joda.time.DateTime
import org.joda.time.Hours.hoursBetween
import org.joda.time.Minutes.minutesBetween
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{Map => JMap}
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._

@Service
class ContextHelper @Autowired()(val serverConfiguration: ServerConfiguration,
                                 val configurationRepository: ConfigurationRepository,
                                 implicit val userInfoResolver: UserInfoResolver) {

  def baseContext: ScalaMustacheContext = {
    Map[String, Any]("url" -> serverConfiguration.getServerUrl) ++ getThemeContext ++ getCustomLogoContext
  }

  def getThemeContext: Map[String, Any] = {
    val themeSettings = configurationRepository.getThemeSettings()
    val headerName = if (themeSettings.getHeaderName == null || themeSettings.getHeaderName.isEmpty) null else s"(${themeSettings.getHeaderName})"
    Map[String, Any]("themeSettings" -> themeSettings, "environmentName" -> headerName)
  }

  def getCustomLogoContext: Map[String, Any] = {
    try {
      configurationRepository.read(CustomLogoSettings.CUSTOM_LOGO_SETTINGS_ID)
      Map[String, Any]("isCustomLogoSettingsDefined" -> true)
    } catch {
      case e: NotFoundException => Map[String, Any]("isCustomLogoSettingsDefined" -> false)
    }
  }

  def releasesContext(releases: List[Release]): JavaMustacheContext = releases match {
    case r :: Nil => (baseContext + ("release" -> getReleaseProperties(r))).asJava
    case _ => (baseContext + ("releases" -> releases.map(getReleaseProperties).asJava)).asJava
  }

  def reportContext(reportJobs: List[ReportJobNotificationTrigger]): JavaMustacheContext = reportJobs match {
    case t :: Nil =>
      val reportProperties = Map(
        "ttlInDays" -> t.ttlInDays.toString,
        "formattedTtlInDays" -> s"${t.ttlInDays.toString} ${if (t.ttlInDays == 1) "day" else "days"}",
        "url" -> t.reportUrl,
        "reportName" -> t.reportName
      ).asJava
      (baseContext + ("report" -> reportProperties)).asJava
    case _ => Map[String, Any]().asJava // bulk context not implemented
  }

  def tasksContext(tasks: List[Task]): JavaMustacheContext = tasks match {
    case t :: Nil => taskContext(t)
    case _ => Map[String, Any]().asJava // bulk context not implemented
  }

  def commentsContext(comment: Comment)(tasks: List[Task]): JavaMustacheContext = tasks match {
    case t :: Nil => commentContext(comment, t)
    case _ => Map[String, Any]().asJava // bulk context not implemented
  }

  def mentionContext(mentionedBy: String, mentionText: String)(tasks: List[Task]): JavaMustacheContext = {
    val resolvedText = resolveMentions(mentionText)
    val mentionProperties = Map(
      "mentionedBy" -> mentionedBy,
      "mentionedByFullName" -> userInfoResolver.getFullNameOrUsernameOf(mentionedBy),
      "mentionText" -> resolvedText,
      "htmlMentionText" -> renderAsMarkdown(resolvedText)
    )
    (tasksContext(tasks).asScala ++ mentionProperties).asJava
  }

  def userTokenContext(triggers: List[UserTokenAboutToExpire]): JavaMustacheContext = triggers match {
    case t :: Nil =>
      val notificationProperties = Map(
        "tokenNote" -> t.tokenNote,
        "expirationDurationInHours" -> t.expirationDurationInHours
      ).asJava
      (baseContext + ("token" -> notificationProperties)).asJava
    case _ => Map[String, Any]().asJava // bulk context not implemented
  }

  def genericNotificationContext(g: List[GenericSystemNotification]): JavaMustacheContext = g match {
    case t :: Nil =>
      val notificationProperties = Map(
        "subject" -> t.subject,
        "body" -> t.body
      ).asJava
      (baseContext + ("notification" -> notificationProperties)).asJava
    case _ => Map[String, Any]().asJava // bulk context not implemented
  }

  private def taskContext(task: Task): JavaMustacheContext = {
    val (hours, minutes) = task.getOrCalculateDueDate().asScala
      .map(date => {
        val now = DateTime.now()
        val dueDate = new DateTime(date)
        (hoursBetween(now, dueDate).getHours, minutesBetween(now, dueDate).getMinutes % 60)
      }).getOrElse((null, null))
    val ownerFullName = Option(userInfoResolver.getFullNameOrUsernameOf(task.getOwner))
    val description = resolveMentions(task.getDescription)
    val taskProperties = task.getType.getDescriptor.getPropertyDescriptors.asScala.map(pd => pd.getName -> pd.get(task)).toMap ++ Map(
      "url" -> s"${serverConfiguration.getServerUrl}#/tasks/${DOMAIN.convertToViewId(task.getId)}?showDetails=true",
      "ownerFullName" -> ownerFullName.getOrElse("nobody"),
      "description" -> description,
      "htmlDescription" -> renderAsMarkdown(description),
      "dueInHours" -> hours,
      "dueInMinutes" -> minutes
    )
    (releasesContext(List(task.getRelease)).asScala + ("task" -> taskProperties.asJava)).asJava
  }

  private def commentContext(comment: Comment, task: Task): JavaMustacheContext = {
    val commentProperties = comment.getType.getDescriptor.getPropertyDescriptors.asScala.map(pd => pd.getName -> pd.get(comment)).toMap ++ Map(
      "authorFullName" -> userInfoResolver.getFullNameOrUsernameOf(comment.getAuthor),
      "text" -> resolveMentions(comment.getText)
    )
    (taskContext(task).asScala + ("comment" -> commentProperties.asJava)).asJava
  }

  private def getReleaseProperties(release: Release): JMap[String, AnyRef] = {
    val releaseProperties = release.getType.getDescriptor.getPropertyDescriptors.asScala.map(pd => pd.getName -> pd.get(release)).toMap
    val releaseUrl = "url" -> s"${serverConfiguration.getServerUrl}#/releases/${DOMAIN.convertToViewId(release.getId)}"

    (releaseProperties + releaseUrl).asJava
  }

  private def renderAsMarkdown(text: String): String = {
    MarkdownProcessor.process(text).asScala.filterNot(_.isEmpty).orNull
  }
}
