package com.xebialabs.xlrelease.notifications

import com.xebialabs.xlrelease.domain.{Comment, Release, Task}
import com.xebialabs.xlrelease.notifications.TriggerType.TriggerType

sealed trait NotificationTrigger {
  val name: TriggerType // stored in the repo and passed from UI
}

case class CommentAdded(tasks: List[Task], comment: Comment) extends NotificationTrigger {
  val name: TriggerType = TriggerType.COMMENT_ADDED
}

case class UserMentioned(tasks: List[Task], mentions: Set[String], mentionedBy: String, mentionText: String) extends NotificationTrigger {
  val name: TriggerType = TriggerType.USER_MENTIONED
}

// Release notifications
sealed trait ReleaseNotificationTrigger extends NotificationTrigger {
  val releases: List[Release]
}

case class ReleaseStarted(override val releases: List[Release]) extends ReleaseNotificationTrigger {
  val name: TriggerType = TriggerType.RELEASE_STARTED
}

case class ReleaseAborted(override val releases: List[Release]) extends ReleaseNotificationTrigger {
  val name: TriggerType = TriggerType.RELEASE_ABORTED
}

case class ReleaseCompleted(override val releases: List[Release]) extends ReleaseNotificationTrigger {
  val name: TriggerType = TriggerType.RELEASE_COMPLETED
}

case class ReleaseFlagged(override val releases: List[Release]) extends ReleaseNotificationTrigger {
  val name: TriggerType = TriggerType.RELEASE_FLAGGED
}

case class ReleaseFailed(override val releases: List[Release]) extends ReleaseNotificationTrigger {
  val name: TriggerType = TriggerType.RELEASE_FAILED
}

case class ReleaseFailing(override val releases: List[Release]) extends ReleaseNotificationTrigger {
  val name: TriggerType = TriggerType.RELEASE_FAILING
}

// Task notifications
sealed trait TaskNotificationTrigger extends NotificationTrigger {
  val tasks: List[Task]
}

case class ActiveTaskAssigned(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.ACTIVE_TASK_ASSIGNED
}

case class ActiveTaskUnassigned(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.ACTIVE_TASK_UNASSIGNED
}

case class ManualTaskStarted(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.MANUAL_TASK_STARTED
}

case class TaskFailed(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.TASK_FAILED
}

case class TaskOverdue(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.TASK_OVERDUE
}

case class TaskWaitingForInput(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.TASK_WAITING_FOR_INPUT
}

case class ManualTaskStartedWithoutAssignee(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.MANUAL_TASK_STARTED_WITHOUT_OWNER
}

case class TaskFlagged(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.TASK_FLAGGED
}

case class TaskDueSoon(override val tasks: List[Task]) extends TaskNotificationTrigger {
  val name: TriggerType = TriggerType.TASK_DUE_SOON
}

// Report Notifications
sealed trait ReportJobNotificationTrigger extends NotificationTrigger {
  val reportUrl: String
  val reportName: String
  val ttlInDays: Integer
  val owner: String
}

case class AuditReportJobCompleted(reportUrl: String, reportName: String, ttlInDays: Integer, owner: String) extends ReportJobNotificationTrigger {
  val name: TriggerType = TriggerType.AUDIT_REPORT_JOB_COMPLETED
}

case class ReportJobFailed(reportName: String, owner: String) extends ReportJobNotificationTrigger {
  val name: TriggerType = TriggerType.AUDIT_REPORT_JOB_FAILED
  override val reportUrl: String = null
  override val ttlInDays: Integer = 0
}

case class ReportJobAborted(reportName: String, owner: String) extends ReportJobNotificationTrigger {
  val name: TriggerType = TriggerType.AUDIT_REPORT_JOB_ABORTED
  override val reportUrl: String = null
  override val ttlInDays: Integer = 0
}


// User Token Notifications
case class UserTokenAboutToExpire(username: String, tokenNote: String, expirationDurationInHours: Long) extends NotificationTrigger {
  val name: TriggerType = TriggerType.USER_TOKEN_ABOUT_TO_EXPIRE
}

// Generic Notifications
case class GenericSystemNotification(subject: String, body: String) extends NotificationTrigger {
  val name: TriggerType = TriggerType.GENERIC_SYSTEM_NOTIFICATION
}
