package com.xebialabs.xlrelease.notifications.actor

import akka.actor.{ActorRef, Props}
import com.xebialabs.xlplatform.cluster.ClusterMode.{FULL, HOT_STANDBY, STANDALONE}
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.actors.initializer.ActorInitializer
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.notifications.email.EmailNotificationsService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Profile
import org.springframework.stereotype.Component

import scala.concurrent.{Await, Promise}

trait NotificationServiceActorInitializer extends ActorInitializer with Logging

@Component
class NotificationServiceActorHolder @Autowired()(xlrConfig: XlrConfig) {
  private val actorPromise = Promise[ActorRef]()

  def actorRef(): ActorRef = {
    Await.result(actorPromise.future, xlrConfig.timeouts.systemInitialization)
  }

  def resolveActorRef(actorRef: ActorRef): Unit = {
    actorPromise.success(actorRef)
  }
}


@Component
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredNotificationServiceActorInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                                   notificationServiceActorHolder: NotificationServiceActorHolder,
                                                                   emailNotificationsService: EmailNotificationsService)
  extends NotificationServiceActorInitializer {
  override def initialize(): Unit = {
    logger.debug("Initializing non-clustered notification service actor...")
    notificationServiceActorHolder.resolveActorRef(
      systemHolder.actorSystem.actorOf(
        Props(new NonClusteredNotificationServiceActor(emailNotificationsService))
      )
    )
  }
}

@Component
@Profile(Array(FULL))
class ClusteredNotificationServiceActorInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                                notificationServiceActorHolder: NotificationServiceActorHolder,
                                                                emailNotificationsService: EmailNotificationsService)
  extends NotificationServiceActorInitializer {
  override def initialize(): Unit = {
    logger.debug("Initializing clustered notification service actor...")
    notificationServiceActorHolder.resolveActorRef(
      systemHolder.actorSystem.actorOf(
        Props(new ClusteredNotificationServiceActor(emailNotificationsService))
      )
    )
  }
}
