package com.xebialabs.xlrelease.notifications.email

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.configuration.CustomLogoSettings
import com.xebialabs.xlrelease.notifications.configuration.{BasicSmtpAuthentication, OAuth2SmtpAuthentication, SmtpServer}
import com.xebialabs.xlrelease.service.ConfigurationService
import grizzled.slf4j.Logging
import org.apache.commons.mail.HtmlEmail
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.nio.charset.StandardCharsets
import java.util.Base64
import javax.activation.{DataSource, URLDataSource}
import javax.mail.util.ByteArrayDataSource

@Component
class EmailFactory @Autowired()(passwordEncrypter: PasswordEncrypter, configurationService: ConfigurationService) extends Logging {

  private val SOCKET_TIMEOUT = 30000

  def newHtmlEmail(smtpServer: SmtpServer): HtmlEmail = {
    val email = new HtmlEmail
    val logo = s"/email-logo-digital-ai-release.png"
    val url = getClass.getResource(logo)
    require(url != null)
    email.embed(new URLDataSource(url), "logo", "logo")

    try {
      val customLogoSettings: CustomLogoSettings = configurationService.read(CustomLogoSettings.CUSTOM_LOGO_SETTINGS_ID).asInstanceOf[CustomLogoSettings]
      val customLogoByteArray = Base64.getDecoder.decode(customLogoSettings.getContent)
      val customLogoDataSource: DataSource = new ByteArrayDataSource(customLogoByteArray, customLogoSettings.getContentType)
      email.embed(customLogoDataSource, "customLogo", "customLogo")
    } catch {
      case e: NotFoundException => debug("Custom logo is not found");
    }

    email.setHostName(smtpServer.host)
    email.setSmtpPort(smtpServer.port)
    email.setStartTLSEnabled(smtpServer.tls)
    email.setSocketConnectionTimeout(SOCKET_TIMEOUT)
    email.setSocketTimeout(SOCKET_TIMEOUT)
    email.setCharset(StandardCharsets.UTF_8.name)

    smtpServer.authentication match {
      case basicSmtpAuthentication: BasicSmtpAuthentication =>
        email.setAuthentication(basicSmtpAuthentication.username, getPassword(basicSmtpAuthentication.password))
      case oauth2SmtpAuthentication: OAuth2SmtpAuthentication =>
        email.setAuthentication(smtpServer.getFromAddress, getPassword(oauth2SmtpAuthentication.getAccessToken))
        val mailProperties = email.getMailSession.getProperties
        mailProperties.setProperty("mail.smtp.auth.mechanisms", "XOAUTH2")
      case _ =>
    }
    email
  }

  private def getPassword(password: String): String = {
    if (passwordEncrypter.isEncoded(password)) {
      passwordEncrypter.decrypt(password)
    } else {
      password
    }
  }
}
