package com.xebialabs.xlrelease.notifications.configuration;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlrelease.domain.BaseConfiguration;
import com.xebialabs.xlrelease.domain.notification.MailPriority;
import com.xebialabs.xlrelease.notifications.configuration.trigger.NotificationTriggerSettings;
import com.xebialabs.xlrelease.notifications.configuration.trigger.RecipientSettings;
import com.xebialabs.xlrelease.utils.Collectors;

import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION;
import static java.util.stream.Collectors.toList;

@Metadata(root = CONFIGURATION, versioned = false)
public class EmailNotificationSettings extends BaseConfiguration {

    public static final String GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID = "Configuration/mail/EmailNotificationSettings";

    @Property(label = "Cardinality", defaultValue = "1", hidden = true)
    private Integer cardinality;

    public Integer getCardinality() {
        return cardinality;
    }

    /**
     * For the value string JSON take a look at {@link NotificationTriggerSettings}
     */
    @Property(description = "Contains information about settings for the raised event")
    private Map<String, String> notificationTriggerSettings = new HashMap<>();

    public boolean hasNotificationTriggerSettings(String notificationTriggerName) {
        return notificationTriggerSettings.containsKey(notificationTriggerName);
    }

    @VisibleForTesting
    public void removeNotificationTriggerSettings(String notificationTriggerName) {
        notificationTriggerSettings.remove(notificationTriggerName);
    }

    public NotificationTriggerSettings getNotificationTriggerSettings(String notificationTriggerName) {
        ObjectMapper objectMapper = newObjectMapper();
        String settingsJson = notificationTriggerSettings.get(notificationTriggerName);
        return readJsonToNotificationTriggerSettings(objectMapper, settingsJson);
    }

    public List<NotificationTriggerSettings> getNotificationTriggerSettings() {
        ObjectMapper objectMapper = newObjectMapper();
        return notificationTriggerSettings.values().stream()
                .map(settingsJson -> readJsonToNotificationTriggerSettings(objectMapper, settingsJson))
                .collect(toList());
    }

    public void setNotificationTriggerSettings(List<NotificationTriggerSettings> notificationTriggerSettings) {
        ObjectMapper objectMapper = newObjectMapper();
        this.notificationTriggerSettings = notificationTriggerSettings.stream()
                .collect(Collectors.toMap(NotificationTriggerSettings::getNotificationTriggerName, settings -> {
                    try {
                        return objectMapper.writeValueAsString(settings);
                    } catch (JsonProcessingException e) {
                        throw new IllegalArgumentException("Serialization failed for NotificationTriggerSettings with name " + settings.getNotificationTriggerName(), e);
                    }
                }));
    }

    public void setNotificationTriggerSettings(NotificationTriggerSettings notificationTriggerSettings) {
        try {
            this.notificationTriggerSettings.put(notificationTriggerSettings.getNotificationTriggerName(),
                    newObjectMapper().writeValueAsString(notificationTriggerSettings));
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Serialization failed for NotificationTriggerSettings with name " +
                    notificationTriggerSettings.getNotificationTriggerName(), e);
        }
    }

    private static ObjectMapper newObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        return objectMapper;
    }

    private static NotificationTriggerSettings readJsonToNotificationTriggerSettings(ObjectMapper objectMapper, String settingsJson) {
        NotificationTriggerSettings settings;

        try {
            settings = objectMapper.readValue(settingsJson, NotificationTriggerSettings.class);
        } catch (IOException | NullPointerException | IllegalArgumentException e) {
            throw new IllegalStateException("Deserialization failed for NotificationTriggerSettings with JSON: " + settingsJson, e);
        }

        if (settings != null) {
            if (settings.getPriority() == null) {
                settings.setPriority(MailPriority.Normal);
            }
            if (settings.getRecipientSettings() == null) {
                settings.setRecipientSettings(new RecipientSettings());
            }
        }
        return settings;
    }
}
