package com.xebialabs.xlrelease.notifications.initialize

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.notifications.configuration.{BasicSmtpAuthentication, NoSmtpAuthentication, SmtpServer}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.common.BaseInitializingUpgrade
import com.xebialabs.xlrelease.utils.CiHelper
import grizzled.slf4j.Logging
import org.codehaus.jettison.json.JSONObject
import org.springframework.beans.factory.annotation.Autowired

class XLRelease1010SmtpServerUpgrader @Autowired()(configurationRepository: ConfigurationRepository,
                                                   configurationPersistence: ConfigurationPersistence)
  extends BaseInitializingUpgrade with Logging {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "10.1.0#3")

  override def doUpgrade(): Boolean = {
    if (configurationRepository.exists(SmtpServer.SMTP_SERVER_ID)) {
      logger.info("Found SMTP configuration. It will be upgraded.")
      try {
        val smtpServerData = configurationPersistence.findById(SmtpServer.SMTP_SERVER_ID).head._2
        val smtpServerJson = new JSONObject(smtpServerData)

        val smtpServer = new SmtpServer
        smtpServer.setHost(smtpServerJson.getString("host"))
        smtpServer.setPort(smtpServerJson.getInt("port"))
        smtpServer.setTls(smtpServerJson.getBoolean("tls"))
        smtpServer.setFromAddress(smtpServerJson.getString("fromAddress"))
        if (smtpServerJson.has("testAddress")) {
          smtpServer.setTestAddress(smtpServerJson.getString("testAddress"))
        }

        val authentication = if (smtpServerJson.has("username") && !isNullOrEmpty(smtpServerJson.getString("username"))) {
          val username = smtpServerJson.getString("username")
          val password = if (smtpServerJson.has("password")) {
            smtpServerJson.getString("password")
          } else {
            PasswordEncrypter.getInstance().ensureEncrypted("")
          }
          val basicSmtpAuthentication = new BasicSmtpAuthentication
          basicSmtpAuthentication.setUsername(username)
          basicSmtpAuthentication.setPassword(password)
          basicSmtpAuthentication
        } else {
          new NoSmtpAuthentication
        }
        smtpServer.setAuthentication(authentication)
        smtpServer.setId(SmtpServer.SMTP_SERVER_ID)
        CiHelper.fixUpInternalReferences(smtpServer)
        configurationRepository.update(smtpServer)
      } catch {
        case e: Exception => logger.error("Unable to upgrade SMTP configuration", e)
      }

    }
    true
  }

}
