package com.xebialabs.xlrelease.notifications.email

import com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty
import com.xebialabs.deployit.security.{Role, RoleService}
import com.xebialabs.xlrelease.api.internal.DecoratorsCache
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Team.RELEASE_ADMIN_TEAMNAME
import com.xebialabs.xlrelease.domain.{PlanItem, Release, Task, Team}
import com.xebialabs.xlrelease.notifications.configuration.trigger.RecipientSettings._
import com.xebialabs.xlrelease.repository.Ids.{findFolderId, isRoot, releaseIdFrom}
import com.xebialabs.xlrelease.service.{PrincipalInfoResolver, TeamService, UserProfileService}
import com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER
import com.xebialabs.xlrelease.variable.VariableHelper
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{Map => JMap}
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._

@Service
class EmailRecipientResolver @Autowired()(roleService: RoleService,
                                          teamService: TeamService,
                                          userProfileService: UserProfileService,
                                          principalInfoResolver: PrincipalInfoResolver,
                                          xlrConfig: XlrConfig) {

  def resolveRecipient(principalName: String): String = principalInfoResolver.getNotificationEmailOf(principalName)

  def resolveRecipients(roles: List[String], globalRoles: List[String], users: List[String],
                        planItem: PlanItem, excludeAuthor: Boolean = false): Set[String] = {
    val assignments = roleService.readRoleAssignments().asScala.toList
    val cache = new DecoratorsCache()
    val allPrincipals = parseRoles(assignments, roles, planItem, cache) ++ parseGlobalRoles(assignments, globalRoles) ++ parseFolderTeams(assignments, roles, planItem, cache) ++ users --
      (if (excludeAuthor) Set(AUTHENTICATED_USER.getName) else Nil)

    allPrincipals
      .filter(notifyUser)
      .map(principalInfoResolver.getNotificationEmailOf)
      .filter(Option(_).exists(_.trim.nonEmpty))
  }

  private def notifyUser(username: String): Boolean = {
    val profile = userProfileService.findByUsername(username)
    if (profile != null) {
      profile.isLoginAllowed || xlrConfig.isNotifyInactiveUsers
    } else {
      true //Set default to true if profile not found which can be an LDAP user or group
    }
  }

  private def parseRoles(assignments: List[Role], roles: List[String], planItem: PlanItem, cache: DecoratorsCache): Set[String] = {
    planItem match {
      case task: Task =>
        parseTaskRoles(task, roles, assignments, cache)
      case release: Release =>
        parseReleaseRoles(release, roles, assignments, cache)
    }
  }

  private def replaceVariable(value: String, variables: JMap[String, String]): String = {
    if (VariableHelper.containsVariables(value)) {
      VariableHelper.replaceAll(value, variables)
    } else {
      value
    }
  }

  private def parseTaskRoles(task: Task, roles: List[String], assignments: List[Role], cache: DecoratorsCache): Set[String] = {
    var parsed = Set.empty[String]
    val shouldSendToTaskOwner = roles.contains(TASK_OWNER_ROLE_NAME)
    val taskOwner = Option(task.getOwner)

    if (shouldSendToTaskOwner && taskOwner.isDefined) {
      parsed += replaceVariable(taskOwner.get, task.getRelease().getAllStringVariableValues())
    }
    if (roles.contains(TASK_TEAM_ROLE_NAME) || shouldSendToTaskOwner && taskOwner.isEmpty) {
      teamService.findTeamByName(releaseIdFrom(task.getId), task.getTeam).asScala.foreach(parsed ++= getAllMembers(_, assignments))
    }
    if (roles.contains(RELEASE_ADMIN_ROLE_NAME)) {
      parsed ++= getReleaseMembers(task.getRelease, assignments, cache) getOrElse Nil
    }
    if (roles.contains(WATCHER_ROLE_NAME)) {
      val watchers = task.getWatchers.asScala.toSet
      watchers.foreach(parsed += replaceVariable(_, task.getRelease().getAllStringVariableValues()))
    }

    parsed
  }

  private def parseReleaseRoles(release: Release, roles: List[String], assignments: List[Role], cache: DecoratorsCache): Set[String] = {
    var parsed = Set.empty[String]

    if (roles.contains(RELEASE_ADMIN_ROLE_NAME)) {
      parsed ++= getReleaseMembers(release, assignments, cache) getOrElse Nil
    }

    parsed
  }

  private def getReleaseMembers(release: Release, assignments: List[Role], cache: DecoratorsCache): Option[Set[String]] = {
    val effectiveTeams = teamService.getEffectiveTeams(release, cache).asScala.find(_.getTeamName == RELEASE_ADMIN_TEAMNAME)
    effectiveTeams.map(getAllMembers(_, assignments))
  }

  private def parseGlobalRoles(assignments: List[Role], globalRoles: List[String]): List[String] = {
    globalRoles match {
      case Nil => List()
      case _ =>
        assignments
          .filter(r => globalRoles.contains(r.getName))
          .flatMap(_.getPrincipals.asScala.toList)
    }
  }

  private def parseFolderTeams(assignments: List[Role], roles: List[String], planItem: PlanItem, cache: DecoratorsCache): Set[String] = {
    var parsed = Set.empty[String]
    val folderTeams = roles diff List(RELEASE_ADMIN_ROLE_NAME, TASK_OWNER_ROLE_NAME, TASK_TEAM_ROLE_NAME, WATCHER_ROLE_NAME)
    val folderId = findFolderId(planItem.getId)

    if (isNotEmpty(folderId) && !isRoot(folderId)) {
      teamService.findTeamsByNames(folderId, folderTeams.asJava, cache).forEach(parsed ++= getAllMembers(_, assignments))
    }

    parsed
  }

  private def getAllMembers(team: Team, assignments: List[Role]): Set[String] = {
    val teamMembers = team.getMembers.asScala.toSet
    val roleMembers = for {
      roleName <- team.getRoles.asScala.toList
      role <- assignments.find(_.getName == roleName).toList
      principal <- role.getPrincipals.asScala.toList
    } yield {
      // Discover and create a UserProfile for external users.
      userProfileService.discover(principal)

      principal
    }

    teamMembers ++ roleMembers
  }
}
