package com.xebialabs.xlrelease.notifications.handlers

import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.status.FlagStatus
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.notifications._
import com.xebialabs.xlrelease.notifications.mentions.Mentions.collectMentions
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.compat.java8.OptionConverters._

@Component
@EventListener
class TaskNotificationEventHandler @Autowired()(notificationService: NotificationService) {

  @AsyncSubscribe
  def onTaskUpdated(event: TaskUpdatedEvent): Unit = {
    val (original, updated) = (event.original, event.updated)

    if (updated.isActive) {
      notifyOwnerAndTeam(event)

      if (original.getFlagStatus != updated.getFlagStatus && updated.getFlagStatus != FlagStatus.OK) {
        notificationService.notify(TaskFlagged(updated))
      }
    }

    notifyMentions(event)
  }

  private def notifyOwnerAndTeam(event: TaskUpdatedEvent): Unit = {
    val (original, updated) = (event.original, event.updated)
    val originalOwner = Option(original.getOwner)
    val updatedOwner = Option(updated.getOwner)

    val originalTeam = Option(original.getTeam)
    val updatedTeam = Option(updated.getTeam)

    def isOwnerOrTeamUnassigned: Boolean = {
      (originalOwner.isDefined && updatedOwner.isEmpty) || (originalTeam.isDefined && updatedTeam.isEmpty)
    }

    def isOwnerUpdated: Boolean = {
      originalOwner != updatedOwner
    }

    def isTeamUpdated: Boolean = {
      originalTeam != updatedTeam
    }

    if (isOwnerOrTeamUnassigned) {
      notificationService.notify(ActiveTaskUnassigned(original))
    } else if (isOwnerUpdated) {
      if (originalOwner.isDefined) {
        notificationService.notify(ActiveTaskUnassigned(original))
      }
      notificationService.notify(ActiveTaskAssigned(updated))
    } else if (isTeamUpdated) {
      if (originalTeam.isDefined) {
        notificationService.notify(ActiveTaskUnassigned(original))
      }
      notificationService.notify(ActiveTaskAssigned(updated))
    }
  }

  private def notifyMentions(event: TaskUpdatedEvent): Unit = {
    val (original, updated) = (event.original, event.updated)
    if (original.getDescription != updated.getDescription) {
      val originalMentions = collectMentions(original.getDescription)
      val updatedMentions = collectMentions(updated.getDescription)

      val newMentions = updatedMentions -- originalMentions
      if (newMentions.nonEmpty) {
        notificationService.notify(UserMentioned(updated, newMentions, event.username, updated.getDescription))
      }
    }
  }

  @AsyncSubscribe
  def onTaskStarted(event: TaskStartedEvent): Unit = {
    notifyManualTaskStarted(event.task)
  }

  @AsyncSubscribe
  def onTaskRetried(event: TaskRetriedEvent): Unit = {
    notifyManualTaskStarted(event.task)
  }

  @AsyncSubscribe
  def onTaskFailed(event: TaskFailedEvent): Unit = {
    if (!event.task.isTaskGroup) {
      notificationService.notify(TaskFailed(event.task))
    }
  }

  @AsyncSubscribe
  def onTaskWaitingForInput(event: TaskWaitingForInputEvent): Unit = {
    notificationService.notify(TaskWaitingForInput(event.task))
  }

  @AsyncSubscribe
  def onTaskOverdue(event: TaskOverdueEvent): Unit = {
    val task = event.task
    val dueDate = task.getOrCalculateDueDate().asScala

    if (task.isInProgress && task.hasStartDate && dueDate.exists(_.after(task.getStartDate))) {
      notificationService.notify(TaskOverdue(task))
    }
  }

  @AsyncSubscribe
  def onTaskDueSoon(event: TaskDueSoonEvent): Unit = {
    notificationService.notify(TaskDueSoon(event.task))
  }

  private def notifyManualTaskStarted(task: Task): Unit = {
    if (!task.isAutomated && !task.isTaskGroup) {
      if (Option(task.getOwner).isEmpty && Option(task.getTeam).isEmpty && task.sendNotificationWhenStarted) {
        notificationService.notify(ManualTaskStartedWithoutAssignee(task))
      } else {
        notificationService.notify(ManualTaskStarted(task))
      }
    }
  }
}
