package com.xebialabs.xlrelease.notifications.email;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import javax.mail.Session;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import org.apache.commons.mail.DefaultAuthenticator;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.HtmlEmail;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.domain.notification.MailPriority;
import com.xebialabs.xlrelease.notifications.configuration.SmtpServer;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.domain.notification.MailPriority.Normal;
import static java.util.Arrays.asList;
import static javax.mail.internet.InternetAddress.parse;

@Component
public class EmailSender {
    private static final Logger logger = LoggerFactory.getLogger(EmailSender.class);

    private ConfigurationRepository configurationRepository;
    private EmailFactory emailFactory;
    private RefreshTokenOAuth2Helper refreshTokenOAuth2Helper;

    @Autowired
    public EmailSender(ConfigurationRepository configurationRepository, EmailFactory emailFactory, RefreshTokenOAuth2Helper refreshTokenOAuth2Helper) {
        this.configurationRepository = configurationRepository;
        this.emailFactory = emailFactory;
        this.refreshTokenOAuth2Helper = refreshTokenOAuth2Helper;
    }

    public CompletableFuture<Void> scheduleEmail(final Email email) {
        CompletableFuture<Void> future = new CompletableFuture<>();
        XlrConfig.getInstance().auxiliaryExecutor().submit(() -> {
            try {
                sendEmailSync(email);
                future.complete(null);
            } catch (Exception e) {
                future.completeExceptionally(e);
            }
        });
        return future;
    }

    public void sendEmailSync(final Email email) throws EmailException, NotFoundException {
        if (email.getRecipients() == null || email.getRecipients().isEmpty()) {
            String msg = String.format("Email with subject '%s' not sent, because the recipient is missing.", email.getSubject());
            logger.warn(msg);
            throw new IllegalArgumentException(msg);
        } else {
            try {
                sendHTML(email);
            } catch (NotFoundException notFoundException) {
                logger.info("Could not send email: {}. Please configure the SMTP Server", notFoundException.toString());
                throw notFoundException;
            } catch (EmailException e) {
                logger.error("Could not send email ", e);
                throw e;
            }
        }
    }

    void sendHTML(Email email) throws EmailException {
        SmtpServer smtpServer = configurationRepository.read(SmtpServer.SMTP_SERVER_ID());
        sendHTML(smtpServer, email);
    }

    public void sendHTML(SmtpServer smtpServer, Email email) throws EmailException {
        boolean overrideCredentials = hasOverriddenCredentials(email);

        if (!overrideCredentials) {
            refreshTokenOAuth2Helper.refreshTokenIfRequired(smtpServer);
        }

        HtmlEmail htmlEmail = emailFactory.newHtmlEmail(smtpServer);

        setRecipients(htmlEmail, email.getRecipients());

        if (email.getFrom().isPresent() && !email.getFrom().get().isEmpty()) {
            setFrom(htmlEmail, email.getFrom().get());
        } else {
            setFrom(htmlEmail, smtpServer.getFromAddress());
        }

        if (overrideCredentials) {
            DefaultAuthenticator authenticator = new DefaultAuthenticator(email.getSenderUsername().get(),
                    PasswordEncrypter.getInstance().ensureDecrypted(email.getSenderPassword().get()));
            Properties mailProps = htmlEmail.getMailSession().getProperties();
            mailProps.setProperty("mail.smtp.auth.mechanisms", "LOGIN PLAIN DIGEST-MD5 NTLM");
            mailProps.setProperty("mail.smtp.auth.xoauth2.disable", "true");

            Session newSession = Session.getInstance(mailProps, authenticator);
            htmlEmail.setMailSession(newSession);
            htmlEmail.setAuthenticator(authenticator);
        }

        setMailPriority(htmlEmail, email.getPriority().orElse(Normal));

        if (email.getBcc().isPresent() && !email.getBcc().get().isEmpty()) {
            htmlEmail.setBcc(toInternetAddresses(email.getBcc().get()));
        }

        if (email.getCc().isPresent() && !email.getCc().get().isEmpty()) {
            htmlEmail.setCc(toInternetAddresses(email.getCc().get()));
        }

        if (email.getReplyTo().isPresent() && !email.getReplyTo().get().isEmpty()) {
            htmlEmail.setReplyTo(toInternetAddresses(newArrayList(email.getReplyTo().get())));
        }

        htmlEmail.setSubject(email.getSubject());
        htmlEmail.setHtmlMsg(email.getBody());
        htmlEmail.send();
    }

    private boolean hasOverriddenCredentials(final Email email) {
        return email.getSenderUsername().isPresent() && !email.getSenderUsername().get().isEmpty() &&
                email.getSenderPassword().isPresent() && !email.getSenderPassword().get().isEmpty();
    }

    void setMailPriority(HtmlEmail htmlEmail, MailPriority mailPriority) {
        Map<String, String> headers = new HashMap<>();
        headers.put("X-Priority", mailPriority.getPriority());
        headers.put("Importance", mailPriority.getImportance());

        htmlEmail.setHeaders(headers);
    }

    @VisibleForTesting
    void setRecipients(HtmlEmail htmlEmail, Collection<String> recipients) throws EmailException {
        htmlEmail.setTo(toInternetAddresses(recipients));
    }

    @VisibleForTesting
    List<InternetAddress> toInternetAddresses(Collection<String> recipients) throws EmailException {
        List<InternetAddress> addresses = newArrayList();
        for (String recipient : recipients) {
            try {
                addresses.addAll(asList(parse(recipient)));
            } catch (AddressException e) {
                throw new EmailException(e);
            }
        }
        return addresses;
    }

    @VisibleForTesting
    void setFrom(HtmlEmail email, String fromAddress) throws EmailException {
        if (fromAddress == null) {
            return;
        }

        try {
            InternetAddress from = new InternetAddress(fromAddress);
            email.setFrom(from.getAddress(), from.getPersonal());
        } catch (AddressException e) {
            throw new EmailException(e);
        }
    }
}
