package com.xebialabs.xlrelease.lookup.api.internal

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.annotation.{JsonDeserialize, JsonSerialize}
import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.deployit.plugin.api.reflect._
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.{CustomScriptTask, PythonScript}
import com.xebialabs.xlrelease.lookup.domain.Parameters
import com.xebialabs.xlrelease.plugins.dashboard.views.TileView
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.serialization.json.jackson.{CiDeserializer, CiSerializer}
import com.xebialabs.xlrelease.service.TaskService
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils.{isMaskedPassword, replacePasswordPropertiesInCiIfNeeded}
import com.xebialabs.xlrelease.views.TaskFullView
import com.xebialabs.xlrelease.views.converters.TasksViewConverter
import org.python.core.PyException
import org.springframework.stereotype.Controller

import java.lang.reflect.InvocationTargetException
import java.util.{List => JList}
import javax.script.ScriptException
import jakarta.ws.rs._
import jakarta.ws.rs.core.MediaType
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters.CollectionHasAsScala

@Path("/lookup")
@Produces(Array(MediaType.APPLICATION_JSON))
@Consumes(Array(MediaType.APPLICATION_JSON))
@Controller
class LookupResource(val taskService: TaskService,
                     val tasksViewConverter: TasksViewConverter,
                     implicit val configurationRepository: ConfigurationRepository,
                     implicit val objectMapper: ObjectMapper) {

  @POST
  def getCompletionData(lookupParam: LookupParam): LookupResult = {
    val desc = lookupParam.ci.getType.getDescriptor
    val ci = lookupParam.ci
    val propertyName = lookupParam.getPropertyName
    val pd: PropertyDescriptor = desc.getPropertyDescriptor(propertyName)
    if (pd == null) {
      throw new IllegalArgumentException(s"Encountered unknown propertyName [$propertyName] for '${desc.getType}'")
    }

    val (methodRef: String, params: Parameters) = getControlTaskData(desc, pd)
    val value = try {
      desc.getControlTask(methodRef).invoke[JList[LookupResultElement]](ci, params)
    } catch {
      case e: IllegalAccessException => throw new DelegateError(null, s"Could not access delegate method [$methodRef]")
      case e: InvocationTargetException => throw new DelegateError(null, s"Could not invoke delegate method [$methodRef]")
      case e: DeployitException => throw e
      case e: RuntimeException if e.getCause.isInstanceOf[DeployitException] => throw e.getCause
      case e: RuntimeException if e.getCause.isInstanceOf[ScriptException] && e.getCause.getCause.isInstanceOf[PyException] =>
        // control tasks are wrapped into RuntimeExceptions in com.xebialabs.deployit.booter.local.utils.ReflectionUtils.handleInvocationTargetException
        throw new DelegateError(cause = null, msg = e.getCause.getCause.getMessage)
      case e => throw new DelegateError(e, s"Error invoking delegate method [$methodRef]")
    }

    LookupResult(value)
  }

  @POST
  @Path("/fullview")
  def getCompletionDataForFullView(lookupTaskParam: LookupTaskParam): LookupResult = {
    val param = new LookupParam()
    param.propertyName = lookupTaskParam.propertyName
    param.ci = tasksViewConverter.toTask(lookupTaskParam.ci) match {
      case x: CustomScriptTask => replacePasswords(x)
      case x => x
    }
    getCompletionData(param)
  }

  @POST
  @Path("/tile")
  def getCompletionDataForTile(lookupTileParam: LookupTileParam): LookupResult = {
    val param = new LookupParam()
    param.propertyName = lookupTileParam.propertyName
    param.ci = TileView.toTile(lookupTileParam.ci)
    getCompletionData(param)
  }


  private def getControlTaskData(desc: Descriptor, pd: PropertyDescriptor) = {
    val propertyName: String = pd.getName
    val inputHint: InputHint = pd.getInputHint
    if (inputHint == null) {
      throw new InputHintConfigurationError(s"Could not find inputHint for [$propertyName]")
    }
    val methodRef = inputHint.getMethodRef
    if (methodRef == null) {
      throw new InputHintConfigurationError(s"Could not find method-ref for [$propertyName]")
    }
    val controlTaskDesc = desc.getControlTask(methodRef)
    if (controlTaskDesc == null) {
      throw new InputHintConfigurationError(s"Encountered unknown method-ref [$methodRef]")
    }
    val parameterType = controlTaskDesc.getParameterObjectType
    val expectedParametersType = Type.valueOf(classOf[Parameters])
    if (parameterType != null && parameterType != expectedParametersType) {
      throw new InputHintConfigurationError(s"parameters-type '$parameterType' for [$methodRef] is not the expected type [$expectedParametersType]")
    }
    val params: Parameters = if (parameterType != null) {
      parameterType.getDescriptor.newInstance("")
    } else {
      new Parameters()
    }
    params.setPropertyName(propertyName)
    (methodRef, params)
  }

  private def replacePasswords(task: CustomScriptTask): PythonScript = {
    if (hasMaskedPasswords(task)) {
      val taskFromDatabase = taskService.getTaskWithoutDecoration(task.getId).asInstanceOf[CustomScriptTask]
      replacePasswordPropertiesInCiIfNeeded(Some(taskFromDatabase.getPythonScript), task.getPythonScript)
    }
    task.getPythonScript
  }

  private def hasMaskedPasswords(task: CustomScriptTask): Boolean = {
    task.getTaskType.getDescriptor.getPropertyDescriptors.asScala.exists { pd =>
      pd.isPassword() && pd.getKind == PropertyKind.STRING && isMaskedPassword(String.valueOf(pd.get(task.getPythonScript)))
    }
  }

}

case class LookupResult(@BeanProperty result: JList[LookupResultElement])

case class LookupResultElement(@BeanProperty label: String, @BeanProperty value: String)

class LookupParam {
  @BeanProperty
  @JsonDeserialize(using = classOf[CiDeserializer])
  @JsonSerialize(using = classOf[CiSerializer])
  var ci: ConfigurationItem = _

  @BeanProperty
  var propertyName: String = _
}

class LookupTaskParam {
  @BeanProperty
  var ci: TaskFullView = _

  @BeanProperty
  var propertyName: String = _
}

class LookupTileParam {
  @BeanProperty
  var ci: TileView = _

  @BeanProperty
  var propertyName: String = _
}


sealed abstract class LookupException(cause: Throwable, msg: String) extends DeployitException(cause, msg)

@HttpResponseCodeResult(statusCode = 500)
class InputHintConfigurationError(msg: String) extends LookupException(null, msg)

@HttpResponseCodeResult(statusCode = 500)
class DelegateError(cause: Throwable, msg: String) extends LookupException(cause, msg)
