package com.xebialabs.xlrelease.actors.kryo.serializers

import com.esotericsoftware.kryo.kryo5.io.{Input, Output}
import com.esotericsoftware.kryo.kryo5.{Kryo, Serializer}
import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.actors.kryo.serializers.DeployitExceptionSerializerSupport.DeployitExceptionData
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.serialization.json.xltype.{CiJson2Reader, CiListJson2Reader}

import java.util

trait DeployitExceptionSerializerSupport[T <: DeployitException] {
  self: Serializer[_ <: T] =>

  def writeDeployitException(kryo: Kryo, output: Output, item: T): Unit = {
    val cause = Option(item.getCause).flatMap(cause =>
      Option(kryo.getClassResolver.getRegistration(cause.getClass))
        .map(_ => item.getCause)
    ).orNull
    val (isList, entityJson) = item.getEntity match {
      case ciCollection: util.Collection[_] =>
        import scala.jdk.CollectionConverters._
        if (ciCollection.asScala.forall(_.isInstanceOf[ConfigurationItem])) {
          true -> CiSerializerHelper.serialize(new util.ArrayList[ConfigurationItem](ciCollection.asInstanceOf[util.Collection[ConfigurationItem]]))
        } else {
          false -> null
        }
      case ci: ConfigurationItem =>
        (false, CiSerializerHelper.serialize(ci))
      case _ =>
        (false, null)
    }
    kryo.writeClassAndObject(output, item.getErrors)
    kryo.writeClassAndObject(output, item.getStackTrace)
    kryo.writeClassAndObject(output, cause)
    // we also have to write a ref somehow, but only if it's a list of cis or a ci, and only via JSON?
    kryo.writeClassAndObject(output, isList)
    kryo.writeClassAndObject(output, entityJson)
  }


  def readDeployitException(kryo: Kryo, input: Input): DeployitExceptionData = {
    val errors = kryo.readClassAndObject(input).asInstanceOf[java.util.List[String]]
    val stackTrace = kryo.readClassAndObject(input).asInstanceOf[Array[StackTraceElement]]
    val cause = kryo.readClassAndObject(input).asInstanceOf[Throwable]
    val isList = kryo.readClassAndObject(input).asInstanceOf[Boolean]
    val entityJson = kryo.readClassAndObject(input).asInstanceOf[String]
    val (entity, entityList) = if (null != entityJson) {
      val ciConverter = new PasswordEncryptingCiConverter()
      ciConverter.setReadValidationMessages(true)
      if (isList) {
        val ciListReader = CiListJson2Reader.create(entityJson)
        val entityList = ciConverter.readCis(ciListReader)
        (null, entityList)
      } else {
        val ciJsonReader = CiJson2Reader.create(entityJson)
        val entity = ciConverter.readCi(ciJsonReader)
        (entity, null)
      }
    } else {
      (null, null)
    }
    DeployitExceptionData(errors, stackTrace, cause, entity, entityList)
  }

}

object DeployitExceptionSerializerSupport {
  case class DeployitExceptionData(errors: java.util.List[String],
                                   stackTrace: Array[StackTraceElement],
                                   cause: Throwable,
                                   entity: ConfigurationItem,
                                   entityList: util.List[ConfigurationItem]
                                  ) {

    import scala.jdk.CollectionConverters._

    def msg: String = if (null != errors) {
      errors.asScala.mkString("\n")
    } else {
      ""
    }
  }
}
