package com.xebialabs.xlrelease.actors.kryo

import com.esotericsoftware.kryo.kryo5.io.{Input, Output}
import com.esotericsoftware.kryo.kryo5.serializers.FieldSerializer
import com.esotericsoftware.kryo.kryo5.{Kryo, Registration, Serializer}
import com.xebialabs.deployit.checks.Checks.{ConditionFailedException, IncorrectArgumentException, MissingArgumentException}
import com.xebialabs.deployit.core.rest.api.PermissionResource.UnknownPermissionException
import com.xebialabs.deployit.core.rest.api.SecurityResource.{UnknownPermissionException => SecResUnknownPermissionException}
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.exception.{NotFoundException, UnknownLabelException, UnknownRevisionException}
import com.xebialabs.deployit.plugin.api.services.ArtifactTransformerException
import com.xebialabs.deployit.repository.{ItemAlreadyExistsException, ItemConflictException, ItemInUseException}
import com.xebialabs.deployit.security.PermissionDeniedException
import com.xebialabs.deployit.security.authentication.{AuthenticationFailureException, UserAlreadyExistsException}
import com.xebialabs.deployit.service.replacement.PlaceHolderStateException
import com.xebialabs.deployit.service.validation.Validator.ValidationsFailedException
import com.xebialabs.plugin.manager.exception.PluginRepositoryException._
import com.xebialabs.plugin.manager.exception.{InvalidPluginVersionPropertiesException, MissingPluginVersionPropertiesException}
import com.xebialabs.plugin.manager.service.PluginNotFoundException
import com.xebialabs.xlrelease.actors.kryo.serializers.DeployitExceptionSerializerSupport.DeployitExceptionData
import com.xebialabs.xlrelease.actors.kryo.serializers._
import com.xebialabs.xlrelease.actors.kryoserializers._
import com.xebialabs.xlrelease.actors.{KryoInitializer, RemoteException}
import com.xebialabs.xlrelease.api.internal.ProfileResource.InvalidPreviousPasswordException
import com.xebialabs.xlrelease.exception._
import com.xebialabs.xlrelease.lookup.api.internal.{DelegateError, InputHintConfigurationError}
import com.xebialabs.xlrelease.notifications.api.internal.InvalidSmtpConfigException
import com.xebialabs.xlrelease.scm.connector.ScmException
import com.xebialabs.xlrelease.security.authentication.TokenExpiredException
import com.xebialabs.xlrelease.service.RestartPhasesException
import com.xebialabs.xlrelease.status.service.script.AutoconfigureConfigurationError
import com.xebialabs.xlrelease.status.service.{EndpointExternalDeploymentError, PatchError, UpdateFiltersError, WebhookSourceFiltersError}
import com.xebialabs.xlrelease.utils.AttachmentUploadException
import com.xebialabs.xlrelease.validation.XlrValidationsFailedException
import com.xebialabs.xlrelease.versioning.TemplateVersioningException

import scala.reflect.ClassTag

object DeployitExceptionSerializers extends KryoInitializer {

  override def postInit(kryo: Kryo): Unit = {
    // special cases:
    kryo.register(classOf[RemoteException], RemoteExceptionSerializer)
    kryo.register(classOf[XlrValidationsFailedException], XlrValidationsFailedExceptionSerializer)
    // deployit exceptions:
    implicit val k = kryo
    registerBasicDeployitException(data => new AttachmentUploadException(data.msg))
    registerBasicDeployitException(data => new DelegateError(data.cause, data.msg))
    registerBasicDeployitException(data => new InputHintConfigurationError(data.msg))
    registerBasicDeployitException(data => new InvalidPreviousPasswordException(data.cause))
    registerBasicDeployitException(data => InvalidSmtpConfigException(data.msg))
    registerBasicDeployitException(data => new LogFriendlyConcurrentModificationException(data.msg))
    kryo.register(classOf[LogFriendlyNotFoundException], LogFriendlyNotFoundExceptionSerializer)
    registerBasicDeployitException(data => new PortAlreadyInUseException(data.msg))
    registerBasicDeployitException(data => new LicenseLimitReachedException(data.msg))
    registerViaFieldSerializer[RestartPhasesException]
    kryo.register(classOf[ScmException], ScmExceptionSerializer)
    kryo.register(classOf[TemplateVersioningException], TemplateVersioningExceptionSerializer)
    registerBasicDeployitException(data => TokenExpiredException(data.msg))

    // xlr-application-pipelines-module:
    registerApplicationPipelinesExceptions

    registerBasicDeployitException(data => ArtifactTransformerException(data.msg))
    registerBasicDeployitException(data => new AuthenticationFailureException(data.msg))

    // Checks:
    registerBasicDeployitException(data => new ConditionFailedException(data.msg))
    registerBasicDeployitException(data => new IncorrectArgumentException(data.msg))
    registerBasicDeployitException(data => new MissingArgumentException(data.msg))

    // NexusPluginRepositoryException:
    registerNexusPluginRepositoryExceptions

    // platform:
    registerPlatformExceptions

    // everything else goes via default deployit exception
    registerBasicDeployitException(data => new DeployitException()) // handle errors list serialization...
  }


  def registerApplicationPipelinesExceptions(implicit kryo: Kryo): Unit = {
    registerBasicDeployitException(data => EndpointExternalDeploymentError(data.msg))
    registerBasicDeployitException(data => AutoconfigureConfigurationError(data.msg))
    registerBasicDeployitException(data => PatchError(data.msg))
    registerBasicDeployitException(data => UpdateFiltersError(data.msg))
    registerBasicDeployitException(data => WebhookSourceFiltersError(data.msg))
  }

  private def registerNexusPluginRepositoryExceptions(implicit kryo: Kryo): Unit = {
    registerViaFieldSerializer[ChecksumMismatch]
    registerViaFieldSerializer[ContentLengthMismatch]
    registerViaFieldSerializer[DownloadArtifactException]
    registerViaFieldSerializer[DownloadChecksumException]
    registerViaFieldSerializer[NoMetadataVersions]
    registerViaFieldSerializer[QueryFailed]
    registerViaFieldSerializer[SearchMetadataFailed]
    registerViaFieldSerializer[TooManyResults]
  }

  private def registerPlatformExceptions(implicit kryo: Kryo): Unit = {
    registerBasicDeployitException(data => new ItemAlreadyExistsException(data.msg))
    registerBasicDeployitException(data => new ItemConflictException(data.msg))
    registerBasicDeployitException(data => new ItemInUseException(data.msg))
    registerBasicDeployitException(data => new NotFoundException(data.msg))
    registerBasicDeployitException(data => new PermissionDeniedException(data.msg))
    registerBasicDeployitException(data => new PlaceHolderStateException(data.msg))
    // PlaceholderScanningFailedException // TODO: no constructor?!!
    registerBasicDeployitException(data => PluginNotFoundException(data.msg))
    // PluginUninstallException // TODO should not contain BYTES!
    registerBasicDeployitException(data => new UnknownLabelException(data.msg))
    registerBasicDeployitException(data => new SecResUnknownPermissionException(data.msg))
    registerBasicDeployitException(data => new UnknownPermissionException(data.msg))
    registerBasicDeployitException(data => new UnknownRevisionException(data.msg))
    // UnsupportedOidcConfigurationException // TODO used by OIDC plugin
    registerBasicDeployitException(data => new UserAlreadyExistsException(data.msg))
    registerBasicDeployitException(data => MissingPluginVersionPropertiesException(data.msg))
    registerBasicDeployitException(data => InvalidPluginVersionPropertiesException(data.msg))
    kryo.register(classOf[ValidationsFailedException], ValidationsFailedExceptionSerializer)
  }

  private def registerBasicDeployitException[T <: DeployitException : ClassTag](create: DeployitExceptionData => T)(implicit kryo: Kryo): Registration = {
    val clazz = implicitly[ClassTag[T]].runtimeClass.asInstanceOf[Class[T]]
    val serializer = new Serializer[T] with DeployitExceptionSerializerSupport[T] {
      override def write(kryo: Kryo, output: Output, item: T): Unit = {
        writeDeployitException(kryo, output, item)
      }

      override def read(kryo: Kryo, input: Input, itemType: Class[_ <: T]): T = {
        val data = readDeployitException(kryo, input)
        val ex = create(data)
        ex.setStackTrace(data.stackTrace)
        ex.initCause(data.cause)
        if (!ex.hasErrors) {
          // only add errors if we did not populate it already
          ex.addAll(data.errors)
        }
        ex
      }
    }
    kryo.register(clazz, serializer)
  }

  private def registerViaFieldSerializer[T <: DeployitException : ClassTag](implicit kryo: Kryo): Registration = {
    val clazz = implicitly[ClassTag[T]].runtimeClass.asInstanceOf[Class[T]]
    kryo.register(clazz, new FieldSerializer(kryo, clazz))
  }

  override def getOrder: Int = 1
}
