package com.xebialabs.xlrelease.plugins.dashboard.service

import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.TemplateLogo
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.HomeTilesSqlBuilder
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.HomeTilesSqlBuilder.{lastEditedByUserAlias, releasesCounterAlias}
import com.xebialabs.xlrelease.repository.TemplateMetadataRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{CATEGORIES, FOLDERS, RELEASES}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{FolderPersistence, PersistenceSupport, Utils}
import com.xebialabs.xlrelease.security.XLReleasePermissions
import com.xebialabs.xlrelease.security.XLReleasePermissions.AUDIT_ALL
import com.xebialabs.xlrelease.status.repository.persistence.LiveDeploymentSchema.LIVE_DEPLOYMENT
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Service

import java.util.Date
import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

case class TemplateHomeTileRow(releaseId: String,
                               releaseTitle: String,
                               folderName: String,
                               folderId: String,
                               folderPath: String,
                               lastEditedByUser: Date,
                               releaseCount: Int)

case class ReleaseHomeTileRow(releaseId: String,
                              releaseTitle: String,
                              releaseStatus: String,
                              startDate: Date,
                              endDate: Date,
                              folderName: String,
                              folderId: String,
                              folderPath: String,
                              lastEditedByUser: Date)

case class AppHomeTileRow(appId: String,
                          appName: String,
                          appEnvironment: String,
                          appCreateDate: Date,
                          folderName: String,
                          folderId: String,
                          folderPath: String)

case class AppHomeTileFolderCount(amount: Int,
                                  folderName: String,
                                  folderId: String,
                                  folderPath: String)


case class WorkflowTileRow(releaseId: String,
                           releaseTitle: String,
                           releaseStatus: String,
                           releaseStartDate: Date,
                           releaseEndDate: Date,
                           folderName: String,
                           folderId: String,
                           folderPath: String,
                           category: String,
                           templateUid: Int)

case class WorkflowUserTile(releaseId: String,
                        releaseTitle: String,
                        releaseStatus: String,
                        releaseStartDate: Date,
                        releaseEndDate: Date,
                        folderName: String,
                        folderId: String,
                        folderPath: String,
                        categories: Seq[String],
                        logoId: Option[String])

case class WorkflowUserGroupedTile(recent: Seq[WorkflowUserTile], running: Seq[WorkflowUserTile])


@Service
class HomeTilesQueryService(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                            @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                            templateMetadataRepository: TemplateMetadataRepository,
                            folderPersistence: FolderPersistence,
                            permissionEnforcer: PermissionEnforcer,
                            roleService: RoleService) extends Utils with PersistenceSupport {

  private val sqlBuilder = new HomeTilesSqlBuilder()

  def findLastEditedTemplates(username: String): Seq[TemplateHomeTileRow] = {

    val templateRowMapper: RowMapper[TemplateHomeTileRow] = (rs, _) => TemplateHomeTileRow(
      rs.getString(RELEASES.RELEASE_ID),
      rs.getString(RELEASES.RELEASE_TITLE),
      rs.getString(FOLDERS.NAME),
      rs.getString(FOLDERS.FOLDER_ID),
      rs.getString(FOLDERS.FOLDER_PATH),
      rs.getTimestamp(lastEditedByUserAlias),
      rs.getInt(releasesCounterAlias)
    )
    val templateQuery = sqlBuilder.lastTemplateEditedQuery(5)
    findMany(sqlQuery(templateQuery, params("username" -> username), templateRowMapper))
  }

  def findLastEditedReleases(username: String): Seq[ReleaseHomeTileRow] = {

    val releaseRowMapper: RowMapper[ReleaseHomeTileRow] = (rs, _) => ReleaseHomeTileRow(
      rs.getString(RELEASES.RELEASE_ID),
      rs.getString(RELEASES.RELEASE_TITLE),
      rs.getString(RELEASES.STATUS),
      rs.getTimestamp(RELEASES.START_DATE),
      rs.getTimestamp(RELEASES.END_DATE),
      rs.getString(FOLDERS.NAME),
      rs.getString(FOLDERS.FOLDER_ID),
      rs.getString(FOLDERS.FOLDER_PATH),
      rs.getTimestamp(lastEditedByUserAlias)
    )

    val releaseQuery = sqlBuilder.lastReleaseEditedQuery(5)
    findMany(sqlQuery(releaseQuery, params("username" -> username), releaseRowMapper))
  }

  def findLastCreatedApplications(): (Seq[AppHomeTileRow], Seq[AppHomeTileFolderCount]) = {
    val appRowMapper: RowMapper[AppHomeTileRow] = (rs, _) => AppHomeTileRow(
      rs.getString(LIVE_DEPLOYMENT.DEPLOYMENT_ID),
      rs.getString(LIVE_DEPLOYMENT.APPLICATION_NAME),
      rs.getString(LIVE_DEPLOYMENT.DESTINATION),
      rs.getTimestamp(LIVE_DEPLOYMENT.LAST_EVENT_TIME),
      rs.getString(FOLDERS.NAME),
      rs.getString(FOLDERS.FOLDER_ID),
      rs.getString(FOLDERS.FOLDER_PATH)
    )

    val appCountRowMapper: RowMapper[AppHomeTileFolderCount] = (rs, _) => AppHomeTileFolderCount(
      rs.getInt(1),
      rs.getString(FOLDERS.NAME),
      rs.getString(FOLDERS.FOLDER_ID),
      rs.getString(FOLDERS.FOLDER_PATH)
    )

    // copied from SqlFolderRepository.findViewableNodesById
    val folders = if (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL)) {
      folderPersistence.findById(Folder.ROOT_FOLDER_ID)
    } else {
      folderPersistence.findByIdHavingPermission(Folder.ROOT_FOLDER_ID,
        XLReleasePermissions.VIEW_FOLDER,
        authenticationToPrincipals(getAuthentication).asScala,
        getUserRoles)
    }

    val allFlatFolders = folders.topDown.map(_.uid)
    if (allFlatFolders.size == 1) {
      // only root folder, no applications for you
      (Seq.empty, Seq.empty)
    } else {
      val appHomeTileRows = findMany(sqlQuery(sqlBuilder.lastCreatedApplications(5, allFlatFolders), params(), appRowMapper))
      val appHomeTileFolderCounts = findMany(sqlQuery(sqlBuilder.countApplicationsByFolder(allFlatFolders), params(), appCountRowMapper))
      (appHomeTileRows, appHomeTileFolderCounts)
    }
  }

  def countApplications(): Int = {
    val query = sqlBuilder.countApplications()
    sqlQuery(
      query,
      params(),
      _.getInt(1)
    ).head
  }

  def findLastWorkflowByUser(username: String): WorkflowUserGroupedTile = {
    val logoCache = new mutable.HashMap[Int, Option[TemplateLogo]]()

    val runningQuery = sqlBuilder.lastWorkflowRunning(50)
    val runningRows = findMany(sqlQuery(runningQuery, params("owner" -> username), workflowRowMapper))
    val running = mapWorkflowRows(runningRows, logoCache, w => w.releaseStartDate)

    val recentQuery = sqlBuilder.lastWorkflowRecent(50)
    val recentRows = findMany(sqlQuery(recentQuery, params("owner" -> username), workflowRowMapper))
    val recent = mapWorkflowRows(recentRows, logoCache, w => w.releaseEndDate).reverse

    WorkflowUserGroupedTile(recent.take(5), running.take(5))
  }

  private def mapWorkflowRows(rows: Seq[WorkflowTileRow],
                              logoCache: mutable.Map[Int, Option[TemplateLogo]],
                              sorter: WorkflowUserTile => Date)(implicit ord: Ordering[Date]): Seq[WorkflowUserTile] = {
    val seq = ListBuffer.empty[WorkflowUserTile]
    val groupedByTemplateUid = rows.groupBy(_.templateUid)
    groupedByTemplateUid.keys.foreach { templateId =>
      val allWorkflowsFromSameTemplate = groupedByTemplateUid(templateId)
      val groupedByReleaseId = allWorkflowsFromSameTemplate.groupBy(_.releaseId)
      groupedByReleaseId.keys.foreach { releaseId =>
        val categories = ListBuffer.empty[String]
        val rows = groupedByReleaseId(releaseId)
        rows.foreach { row =>
          if (row.category != null) {
            categories += row.category
          }
        }

        if (!logoCache.contains(templateId)) {
          val templateMetadata = templateMetadataRepository.find(templateId)
          val logo = if (templateMetadata.nonEmpty) {
            Option(templateMetadata.get.templateLogo)
          } else {
            None
          }
          logoCache.put(templateId, logo)
        }
        val templateLogo: Option[TemplateLogo] = logoCache(templateId)
        val head = rows.head
        seq += workflowRowToTile(head, templateLogo, categories)
      }
    }

    seq.toSeq.sortBy(sorter)
  }

  private val workflowRowMapper: RowMapper[WorkflowTileRow] = (rs, _) => WorkflowTileRow(
    rs.getString(RELEASES.RELEASE_ID),
    rs.getString(RELEASES.RELEASE_TITLE),
    rs.getString(RELEASES.STATUS),
    rs.getTimestamp(RELEASES.START_DATE),
    rs.getTimestamp(RELEASES.END_DATE),
    rs.getString(FOLDERS.NAME),
    rs.getString(FOLDERS.FOLDER_ID),
    rs.getString(FOLDERS.FOLDER_PATH),
    rs.getString(CATEGORIES.CATEGORY),
    rs.getInt(RELEASES.CI_UID)
  )

  private def workflowRowToTile(row: WorkflowTileRow, logo: Option[TemplateLogo], categories: mutable.Buffer[String]): WorkflowUserTile = {
    WorkflowUserTile(
      row.releaseId,
      row.releaseTitle,
      row.releaseStatus,
      row.releaseStartDate,
      row.releaseEndDate,
      row.folderName,
      row.folderId,
      row.folderPath,
      categories.toSeq,
      logo.map(_.getId)
    )
  }

  private def getUserRoles = {
    roleService.getRolesFor(getAuthentication).asScala.map(_.getId)
  }

}
