import React, { ReactNode } from 'react';
import { DotCardContent, DotIllustration, DotTypography } from '@digital-ai/dot-components';
import { TextAreaMarkdown } from '@xlr-ui/app/react/components/markdown/textarea-markdown.component';
import { TileButtonsRow } from './tile-buttons-row.component';
import { CommonTileHeader } from './common-tile-header.component';
import './common-card-tile.component.less';

export type ButtonType = 'destructive' | 'primary' | 'outlined' | 'text';

export interface ButtonWithLink {
    buttonCaption: string;
    buttonType: ButtonType;
    'data-testid'?: string;
    onClick: () => void;
}

export interface CommonCardTileComponentProps {
    description: string;
    headerEndContent?: ReactNode;
    illustrationId?: string;
    imageLink?: string;
    isMarkdown: boolean;
    primaryButton: ButtonWithLink;
    secondaryButton?: ButtonWithLink;
    subtitle?: string;
    tertiaryButton?: ButtonWithLink;
    title: string;
}

export const MAX_DESCRIPTION_SIZE = 13_000;
export const MAX_DESCRIPTION_SIZE_WARNING = `The text is longer than ${MAX_DESCRIPTION_SIZE} characters, can't be displayed`;

export const CommonCardTileComponent = ({
    description,
    headerEndContent,
    illustrationId,
    imageLink,
    isMarkdown,
    primaryButton,
    secondaryButton,
    subtitle,
    tertiaryButton,
    title,
}: CommonCardTileComponentProps) => {
    const descriptionSanitized = isMarkdown && description.length > MAX_DESCRIPTION_SIZE ? MAX_DESCRIPTION_SIZE_WARNING : description;

    return (
        <div className="card-tile">
            <div>
                <CommonTileHeader endContent={headerEndContent} subtitle={subtitle} title={title} />
                <DotCardContent className="card-tile-content">
                    {isMarkdown ? (
                        <TextAreaMarkdown
                            inputId="card-tile-input"
                            isReadOnly={true}
                            maxRows={Number.MAX_SAFE_INTEGER}
                            name="card-tile"
                            rows={5}
                            text={descriptionSanitized}
                        />
                    ) : (
                        <DotTypography variant="body1">{description}</DotTypography>
                    )}
                    {illustrationId && <DotIllustration illustrationId={illustrationId} />}
                    {imageLink && !illustrationId && <img alt="tile image" src={imageLink}></img>}
                </DotCardContent>
            </div>
            <TileButtonsRow primaryButton={primaryButton} secondaryButton={secondaryButton} tertiaryButton={tertiaryButton} />
        </div>
    );
};
