import React from 'react';
import moment from 'moment';
import { DotIconButton, DotMenu } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { HomeTemplate } from './active-template-tile.component';
import { menuItemEdit, menuItemRelease, TemplateActionsButton, TemplateActionsButtonProps } from './template-actions-button';
import { MenuItem } from '@xlr-ui/app/react/components/menu-item/menu-item.component';
import { DATE_FORMAT_DAY_FIRST } from '@xlr-ui/app/js/locale/constants';
import * as angularAccessor from '@xlr-ui/app/features/common/services/angular-accessor';

describe('TemplateActionsButton', () => {
    const dispatch = jest.fn();

    const onOptionsClick = jest.fn();
    const selected = false;
    const template: HomeTemplate = {
        releaseId: 'releaseId',
        releaseTitle: 'Test Release',
        folderName: 'test',
        folderId: 'folderId',
        folderPath: '/',
        lastEditedByUser: 1688372003150,
        releaseCount: 0,
    };
    const urlTemplateId = 'folderId-releaseId';

    const defaultProps: TemplateActionsButtonProps = { onOptionsClick, selected, template, urlTemplateId };

    const defaultState = {
        profile: { dateFormat: DATE_FORMAT_DAY_FIRST },
    };

    const mount = (props: TemplateActionsButtonProps = defaultProps, state = defaultState) => {
        return mountWithStoreAndTheme(<TemplateActionsButton {...props} />, dispatch, state);
    };

    const getIconButton = (wrapper: ReactWrapper) => wrapper.find(DotIconButton);
    const getMenu = (wrapper: ReactWrapper) => wrapper.find(DotMenu);
    const getReleaseMenuItem = (wrapper: ReactWrapper) => wrapper.find(MenuItem).at(0);
    const getEditMenuItem = (wrapper: ReactWrapper) => wrapper.find(MenuItem).at(1);

    beforeEach(() => {
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        const dateFilterWrapper = (_filter: string) => (date: string, angularJsFormat: string) => moment(date).format(angularJsFormat);
        getAngularServiceSpy.mockReturnValue(dateFilterWrapper);
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render with correct props', () => {
        const wrapper = mount();
        const iconButtonProps = getIconButton(wrapper).props();
        expect(iconButtonProps.iconId).toBe('options');
        const menuProps = getMenu(wrapper).props();
        expect(menuProps.dense).toBe(true);
        expect(menuProps.disablePortal).toBe(true);
        expect(menuProps.id).toBe(template.releaseId);
        expect(menuProps.menuItems).toHaveLength(2);
        expect(menuProps.menuPlacement).toBe('left');
        expect(menuProps.open).toBe(selected);
    });

    it('should show correct menu items', () => {
        const wrapper = mount();
        getIconButton(wrapper).invoke('onClick')?.({ stopPropagation: jest.fn() } as never);

        const releaseMenuItemProps = getReleaseMenuItem(wrapper).props();
        expect(releaseMenuItemProps.iconId).toBe('play');
        expect(releaseMenuItemProps.label).toBe('Start release');
        const editMenuItemProps = getEditMenuItem(wrapper).props();
        expect(editMenuItemProps.iconId).toBe('edit');
        expect(editMenuItemProps.label).toBe('Edit template');
    });

    it('should call onOptionsClick when clicking on options icon button', () => {
        const wrapper = mount();
        getIconButton(wrapper).invoke('onClick')?.({ stopPropagation: jest.fn() } as never);
        expect(onOptionsClick).toBeCalled();
    });

    it('should redirect to proper url when clicking menu item "Start release"', () => {
        const wrapper = mount();
        getIconButton(wrapper).invoke('onClick')?.({ stopPropagation: jest.fn() } as never);

        const menu = getMenu(wrapper);
        menu.invoke('onSelect')?.({ stopPropagation: jest.fn() } as never, 'releaseId', menuItemRelease);
        expect(window.location.href).toBe(`http://localhost/#/releases/create?fromHome=true&fromTemplateId=${urlTemplateId}`);
    });

    it('should redirect to proper url when clicking menu item "Edit"', () => {
        const wrapper = mount();
        getIconButton(wrapper).invoke('onClick')?.({ stopPropagation: jest.fn() } as never);

        const menu = getMenu(wrapper);
        menu.invoke('onSelect')?.({ stopPropagation: jest.fn() } as never, 'releaseId', menuItemEdit);
        expect(window.location.href).toBe(`http://localhost/#/templates/${urlTemplateId}`);
    });
});
