import React from 'react';
import { DotButton, DotCardFooter, DotDialog } from '@digital-ai/dot-components';
import constant from 'lodash/constant';
import { mountWithTheme, ReactWrapper, act } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { WelcomeTile, WelcomeTileProps } from './welcome-tile.component';
import { CommonCardTileComponent } from '../../CommonCardTile/common-card-tile.component';

describe('WelcomeTile', () => {
    let wrapper: ReactWrapper;
    const description = 'my description';
    const title = 'title';

    const defaultProps: WelcomeTileProps = { description, title };

    const mount = async (props: WelcomeTileProps = defaultProps) => {
        await act(async () => {
            wrapper = mountWithTheme(<WelcomeTile {...props} />);
        });
    };

    const getButton = () => wrapper.find(DotCardFooter).find(DotButton);
    const getCommonCardTileComponent = () => wrapper.find(CommonCardTileComponent);
    const getDialog = () => wrapper.find(DotDialog);

    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const togglePendo = (enabled: boolean) => (window.pendo = { isSendingEvents: constant(enabled) });

    afterEach(() => {
        togglePendo(false);
    });

    it('should render', async () => {
        await mount();
        const commonCard = getCommonCardTileComponent();
        const commonCardProps = commonCard.props();
        expect(commonCardProps.title).toBe(title);
        expect(commonCardProps.secondaryButton).toBeUndefined();
        expect(commonCardProps.primaryButton.buttonType).toBe('primary');
        expect(commonCardProps.primaryButton.buttonCaption).toBe('Take quick tour');
        expect(commonCardProps.primaryButton['data-testid']).toBe('welcome-tile-btn');
        expect(commonCardProps.description).toBe(description);
        expect(commonCardProps.imageLink).toBe('static/24.3.7-511.2018/styles/img/pana.svg');
        expect(commonCardProps.isMarkdown).toBe(true);
    });

    it('should do nothing if pendo is enabled', async () => {
        togglePendo(true);
        await mount();
        const button = getButton();
        button.invoke('onClick')?.('' as never);

        expect(getDialog()).not.toExist();
    });

    it('should show dialog on click when pendo is not enabled', async () => {
        togglePendo(false);

        await mount();
        const button = getButton();
        button.invoke('onClick')?.('' as never);

        const dialog = getDialog();
        expect(dialog).toIncludeText(
            'You are seeing this message because your Release product setup has disabled guides and in-product analytics, or you are in an air-gapped environment. Please contact your Release administrator for further assistance.',
        );
        const dialogProps = dialog.props();
        expect(dialogProps.cancelButtonVisible).toBe(false);
        expect(dialogProps.closeIconVisible).toBe(true);
        expect(dialogProps.closeOnClickAway).toBe(false);
        expect(dialogProps.open).toBe(true);
        expect(dialogProps.submitButtonProps).toStrictEqual({
            label: 'Close',
        });
        expect(dialogProps.title).toBe('Content not available');

        dialog.invoke('onSubmit')?.('' as never);
        expect(getDialog()).not.toExist();
    });
});
