import React, { ReactNode } from 'react';
import noop from 'lodash/noop';
import moment from 'moment';
import { DotAvatar, DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { ButtonWithLink } from '../../CommonCardTile/common-card-tile.component';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { WorkflowTileToggleEnum } from './workflow-tile.component';
import { calculateEmptyDataInfo, calculateTableColumns } from './helpers/helpers';
import { HomeWorkflow, workflow as workflowReducer } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getProfile } from '../../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.selectors';
import { RegionalSettings } from '../../../../../../../../../../core/xlr-ui/app/js/locale/regional-settings';
import { EmptyData } from './empty-data.component';
import { getReleaseIdForUrl } from '../../TemplateTile/components/helpers/helpers';
import { ChipGroup } from '../../../../../../../../../../core/xlr-ui/app/react/components/chip-group/chip-group.component';
import { ReleaseStatus } from '../../../../../../../../../../core/xlr-ui/app/features/release-overview/release-status-label/release-status.component';
import { calculateDotDateFormat } from '../../../../../../../../../../core/xlr-ui/app/features/common/helper/date';
import './active-workflow-tile.component.less';

export interface ActiveWorkflowTileProps {
    headerEndContent?: ReactNode;
    primaryButton: ButtonWithLink;
    secondaryButton?: ButtonWithLink;
    subtitle: string;
    tableType: WorkflowTileToggleEnum;
    tertiaryButton?: ButtonWithLink;
    title: string;
    workflows: Array<HomeWorkflow>;
}

const { init } = workflowReducer.actions;

export const ActiveWorkflowTile = ({
    headerEndContent,
    primaryButton,
    secondaryButton,
    subtitle,
    tableType,
    tertiaryButton,
    title,
    workflows,
}: ActiveWorkflowTileProps) => {
    const dispatch = useAppDispatch();
    const { dateFormat: profileDateFormat } = useAppSelector(getProfile);
    const dateFormat = calculateDotDateFormat(RegionalSettings.getDateFormats(profileDateFormat).shortDate);

    const calculateWorkflowsData = () =>
        workflows.map((workflow) => {
            const rowData: Record<string, JSX.Element> = {};
            const workflowStatus = workflow.releaseStatus ? <ReleaseStatus status={workflow.releaseStatus} /> : <></>;

            rowData.workflowName = (
                <div className="workflow-name">
                    {workflow.logoId ? (
                        <DotAvatar alt={workflow.releaseTitle} imageSrc={`api/v1/templates/logo/${workflow.logoId}`} size="small" type="image" />
                    ) : (
                        <DotAvatar alt={workflow.releaseTitle} iconId={'workflow'} size="small" type="icon" />
                    )}
                    <DotTypography data-testid="row-workflow-title" variant="body1">
                        {workflow.releaseTitle}
                    </DotTypography>
                </div>
            );
            rowData.category = <ChipGroup chipCharactersLimit={15} key={tableType} labels={workflow.categories} />;

            if (tableType === WorkflowTileToggleEnum.Running) {
                rowData.status = workflowStatus;
                rowData.started = (
                    <DotTooltip placement="bottom-start" title={moment(workflow.releaseStartDate).format(dateFormat)}>
                        <DotTypography variant="body1">{workflow.releaseStartDate && moment(workflow.releaseStartDate).fromNow()}</DotTypography>
                    </DotTooltip>
                );
            } else if (tableType === WorkflowTileToggleEnum.Recent) {
                rowData.status = workflowStatus;
                rowData.ended = (
                    <DotTooltip placement="bottom-start" title={moment(workflow.releaseEndDate).format(dateFormat)}>
                        <DotTypography variant="body1">{workflow.releaseEndDate && moment(workflow.releaseEndDate).fromNow()}</DotTypography>
                    </DotTooltip>
                );
            } else {
                rowData.numberOfExecutions = (
                    <DotTypography data-testid="row-workflow-executions" variant="body1">
                        {workflow.numberOfExecutions}
                    </DotTypography>
                );
            }

            return {
                id: tableType === WorkflowTileToggleEnum.MostPopular ? workflow.releaseTitle : getReleaseIdForUrl(workflow),
                rowData,
            };
        });

    const onRowClick = (id: string) => {
        if (tableType !== WorkflowTileToggleEnum.MostPopular) {
            window.location.href = `#/stepper/${id}`;
        } else {
            dispatch(init({ searchInput: id }));
        }
    };

    const mainContent =
        workflows.length > 0 ? (
            <DotTable
                className="active-workflow-table"
                columns={calculateTableColumns(tableType)}
                data={calculateWorkflowsData()}
                onRowClick={(_event, id) => onRowClick(id)}
                onUpdateData={noop}
                sortable={false}
            />
        ) : (
            <EmptyData {...calculateEmptyDataInfo(tableType)} />
        );

    return (
        <div className="active-workflow-tile">
            <CommonActiveTile
                avatarIcon="workflow"
                content={mainContent}
                headerEndContent={headerEndContent}
                primaryButton={primaryButton}
                secondaryButton={secondaryButton}
                subtitle={subtitle}
                tertiaryButton={tertiaryButton}
                title={title}
            />
        </div>
    );
};
