import React from 'react';
import classNames from 'classnames';
import moment from 'moment';
import { DotIcon, DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { ButtonWithLink } from '../../CommonCardTile/common-card-tile.component';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { getReleaseIdForUrl } from '../../TemplateTile/components/helpers/helpers';
import { formatReleaseDuration, isReleaseOverdue } from './helpers/helpers';
import { useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getProfile } from '../../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.selectors';
import { getDaysHoursMinutesFromMS } from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/scheduling/helper';
import { RegionalSettings } from '../../../../../../../../../../core/xlr-ui/app/js/locale/regional-settings';
import { ReleaseStatus } from '../../../../../../../../../../core/xlr-ui/app/features/release-overview/release-status-label/release-status.component';
import { ReleaseStatus as ReleaseStatusType } from '../../../../../../../../../../core/xlr-ui/app/types/';
import { calculateDotDateFormat } from '../../../../../../../../../../core/xlr-ui/app/features/common/helper/date';
import './active-release-tile.component.less';

export interface HomeRelease {
    folderId: string;
    folderName: string;
    folderPath: string;
    lastEditedByUser: number;
    releaseEndDate: number;
    releaseId: string;
    releaseStartDate: number;
    releaseStatus: ReleaseStatusType;
    releaseTitle: string;
}

export interface ActiveReleaseTileProps {
    primaryButton: ButtonWithLink;
    releases: Array<HomeRelease>;
    secondaryButton?: ButtonWithLink;
    title: string;
}

export const ActiveReleaseTile = ({ primaryButton, releases, secondaryButton, title }: ActiveReleaseTileProps) => {
    const { dateFormat: profileDateFormat } = useAppSelector(getProfile);
    const dateFormat = calculateDotDateFormat(RegionalSettings.getDateFormats(profileDateFormat).shortDate);

    const tableColumns = [
        { id: 'releases', label: 'Release', truncate: true },
        { id: 'status', label: 'Status', width: '140px' },
        { id: 'endDate', label: 'End date', width: '160px' },
        { id: 'duration', label: 'Duration', width: '140px' },
    ];

    const releasesData = releases.map((release) => {
        const releaseId = getReleaseIdForUrl(release);
        const isReleaseEndDateOverdue = isReleaseOverdue(release);

        const [durationDays, durationHours, durationMinutes] = getDaysHoursMinutesFromMS(release.releaseEndDate - release.releaseStartDate);

        return {
            id: releaseId,
            rowData: {
                releases: (
                    <>
                        <DotTypography data-testid="row-release-title" variant="body1">
                            {release.releaseTitle}
                        </DotTypography>
                        <DotTypography className="release-folder-name" variant="body2">
                            {release.folderName}
                        </DotTypography>
                    </>
                ),
                status: <ReleaseStatus status={release.releaseStatus} />,
                endDate: (
                    <>
                        {isReleaseEndDateOverdue && <DotIcon className="overdue" iconId="delay" />}
                        <DotTooltip data-testid="end-date-tooltip" placement="bottom-start" title={moment(release.releaseEndDate).format(dateFormat)}>
                            <DotTypography className={classNames({ overdue: isReleaseEndDateOverdue })} variant="body1">
                                {moment(release.releaseEndDate).fromNow()}
                            </DotTypography>
                        </DotTooltip>
                    </>
                ),
                duration: (
                    <DotTypography data-testid="row-release-duration" variant="body1">
                        {formatReleaseDuration(durationDays, durationHours, durationMinutes)}
                    </DotTypography>
                ),
            },
        };
    });

    const onRowClick = (id: string) => {
        window.location.href = `#/releases/${id}`;
    };

    const mainContent = (
        <DotTable className="active-release-table" columns={tableColumns} data={releasesData} onRowClick={(_event, id) => onRowClick(id)} sortable={false} />
    );

    return (
        <div className="active-release-tile">
            <CommonActiveTile
                avatarIcon="release"
                content={mainContent}
                primaryButton={primaryButton}
                secondaryButton={secondaryButton}
                subtitle="Last updated - Most recent 5"
                title={title}
            />
        </div>
    );
};
