import React, { useEffect, useState } from 'react';
import classNames from 'classnames';
import { ButtonToggleOption, DotButtonToggle, DotInputSelect, InputSelectOption } from '@digital-ai/dot-components';
import { ButtonWithLink, CommonCardTileComponent } from '../../CommonCardTile/common-card-tile.component';
import { getWorkflowState, HomeWorkflow, workflow } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { ActiveWorkflowTile } from './active-workflow-tile.component';
import { ActiveTileSkeleton } from '../../CommonCardTile/active-tile.skeleton';
import { calculateWorkflowTileSubtitle, mapToHomeWorkflow } from './helpers/helpers';

export enum WorkflowTileToggleEnum {
    MostPopular = 'Most popular',
    Overview = 'Overview',
    Recent = 'Recent',
    Running = 'Running',
}

const workflowsImgSrc = 'static/24.1.8/styles/img/represent_the_stepper.svg';

export interface WorkflowTileProps {
    description: string;
    title: string;
}

const { init, loadWorkflowTile } = workflow.actions;

export const WorkflowTile = ({ description, title }: WorkflowTileProps) => {
    const dispatch = useAppDispatch();
    const { isLoadingWorkflowTile, workflowTile } = useAppSelector(getWorkflowState);
    const showActiveTile = workflowTile.activeTileData.recent.length > 0 || workflowTile.activeTileData.running.length > 0;

    const [toggleValue, setToggleValue] = useState(WorkflowTileToggleEnum.Overview);

    useEffect(() => {
        dispatch(loadWorkflowTile());
    }, []);

    useEffect(() => {
        showActiveTile && setToggleValue(WorkflowTileToggleEnum.MostPopular);
    }, [showActiveTile]);

    const handleOpenWorkflowCatalogClick = () => {
        dispatch(init());
    };

    const handleCreateWorkflowClick = () => {
        window.location.href = '#/workflow-templates';
    };

    const handleManageWorkflowsClick = () => {
        window.location.href = '#/workflow-releases';
    };

    const getWorkflowsForActiveTable = (): Array<HomeWorkflow> => {
        switch (toggleValue) {
            case WorkflowTileToggleEnum.Recent:
                return workflowTile.activeTileData.recent;
            case WorkflowTileToggleEnum.Running:
                return workflowTile.activeTileData.running;
            default:
                return mapToHomeWorkflow(workflowTile.mostPopular);
        }
    };

    const openWorkflowCatalogButton: ButtonWithLink = {
        buttonCaption: 'Open workflow catalog',
        onClick: handleOpenWorkflowCatalogClick,
        buttonType: 'outlined',
    };

    const createWorkflowButton: ButtonWithLink = {
        buttonCaption: 'Create workflow',
        onClick: handleCreateWorkflowClick,
        buttonType: 'text',
    };

    const manageWorkflowsButton: ButtonWithLink = {
        buttonCaption: 'Manage',
        onClick: handleManageWorkflowsClick,
        buttonType: 'text',
    };

    const workflowTileToggleOptions: ButtonToggleOption[] = showActiveTile
        ? [
              {
                  ariaLabel: 'most-popular',
                  text: 'Most popular',
                  value: WorkflowTileToggleEnum.MostPopular,
              },
              {
                  ariaLabel: 'recent',
                  text: 'Recent',
                  value: WorkflowTileToggleEnum.Recent,
              },
              {
                  ariaLabel: 'running',
                  text: 'Running',
                  value: WorkflowTileToggleEnum.Running,
              },
          ]
        : [
              {
                  ariaLabel: 'overview',
                  text: 'Overview',
                  value: WorkflowTileToggleEnum.Overview,
              },
              {
                  ariaLabel: 'most-popular',
                  text: 'Most popular',
                  value: WorkflowTileToggleEnum.MostPopular,
              },
          ];

    const workflowTileSelectOptions: InputSelectOption[] = showActiveTile
        ? [
              { option: 'Most popular', value: WorkflowTileToggleEnum.MostPopular },
              { option: 'Recent', value: WorkflowTileToggleEnum.Recent },
              { option: 'Running', value: WorkflowTileToggleEnum.Running },
          ]
        : [
              { option: 'Overview', value: WorkflowTileToggleEnum.Overview },
              { option: 'Most popular', value: WorkflowTileToggleEnum.MostPopular },
          ];

    const tileHeaderEndContent = (
        <>
            <DotButtonToggle
                buttonOptions={workflowTileToggleOptions}
                className={classNames('workflow-toggle-buttons', { 'with-bottom-margin': toggleValue !== WorkflowTileToggleEnum.Overview })}
                exclusive
                onChange={(_event, buttonValue) => buttonValue && setToggleValue(buttonValue as WorkflowTileToggleEnum)}
                size="small"
                value={toggleValue}
            />
            <DotInputSelect
                className={classNames('workflow-input-select', { 'with-bottom-margin': toggleValue !== WorkflowTileToggleEnum.Overview })}
                id="workflow-input-select"
                name="workflow-input-select"
                onChange={(event) => setToggleValue(event.target.value as WorkflowTileToggleEnum)}
                options={workflowTileSelectOptions}
                value={toggleValue}
            />
        </>
    );

    const workflows = getWorkflowsForActiveTable();
    const hasWorkflowsOnScreen = workflows.length;

    const renderTile = () => {
        return showActiveTile || (!showActiveTile && toggleValue === WorkflowTileToggleEnum.MostPopular) ? (
            <ActiveWorkflowTile
                headerEndContent={tileHeaderEndContent}
                primaryButton={hasWorkflowsOnScreen ? manageWorkflowsButton : createWorkflowButton}
                secondaryButton={hasWorkflowsOnScreen ? createWorkflowButton : openWorkflowCatalogButton}
                subtitle={calculateWorkflowTileSubtitle(toggleValue, workflowTile.archivingSettings?.releaseAgeToDeleteFromJcr)}
                tableType={toggleValue}
                tertiaryButton={hasWorkflowsOnScreen ? openWorkflowCatalogButton : undefined}
                title="Workflow executions"
                workflows={workflows}
            />
        ) : (
            <CommonCardTileComponent
                description={description}
                headerEndContent={tileHeaderEndContent}
                imageLink={workflowsImgSrc}
                isMarkdown={true}
                primaryButton={createWorkflowButton}
                secondaryButton={openWorkflowCatalogButton}
                title={title}
            />
        );
    };

    return isLoadingWorkflowTile ? <ActiveTileSkeleton /> : renderTile();
};
