package com.xebialabs.xlrelease.dsl

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.dsl.service.DslError
import com.xebialabs.xlrelease.repository.CiProperty
import com.xebialabs.xlrelease.variable.VariableFactory
import groovy.transform.TypeChecked

import static com.xebialabs.xlrelease.variable.VariableHelper.isGlobalOrFolderVariable

@TypeChecked
class TaskSpec<T extends Task> extends PlanItemSpec<T> {

  List<VariableRefSpec> variableReferences = []

  // TODO attachments

  TaskSpec(String title = null) {
    this(title, Type.valueOf(Task))
  }

  TaskSpec(String title, final Type ciType) {
    super(title ?: "Task", ciType)
    status(TaskStatus.PLANNED)
  }

  /**
   * Configures facets.
   */
  void facets(@DelegatesTo(value = FacetDelegate, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    def facets = FacetDelegate.delegate(new FacetDelegate(configurationItem), cl)
    configurationItem.getFacets().addAll(facets)
  }

  /**
   * The comments on the task.
   */
  void comments(@DelegatesTo(value = CommentsDelegate, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    configurationItem.comments.addAll(CommentsDelegate.delegate(new CommentsDelegate(), cl))
  }

  /**
   * The state the task is in.
   */
  void status(TaskStatus status) {
    configurationItem.status = status
  }

  /**
   * The tags of the task. Tags can be used for grouping and querying.
   */
  void tags(String... tags) {
    configurationItem.setTags((tags as Set) as List)
  }

  /**
   * The name of the team this task is assigned to.
   */
  void team(String team) {
    configurationItem.team = team
  }

  /**
   * A snippet of code that is evaluated when the task is started.
   */
  void precondition(String precondition) {
    configurationItem.precondition = precondition
  }

  /**
   * Add attachments.
   */
  void attachments(@DelegatesTo(value = ReleaseAttachmentsDelegate, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    configurationItem.attachments = TaskAttachmentsDelegate.delegate(new TaskAttachmentsDelegate(configurationItem), cl)
    configurationItem.release.attachments.addAll(configurationItem.attachments)
  }

  /**
   * Is the task locked.
   */
  void locked(boolean isLocked) {
    configurationItem.locked = isLocked
  }

  @NoDoc
  void variableMapping(Map<String, String> map) {
    configurationItem.variableMapping = map
  }

  /**
   * Utility function that creates a reference to the release variable.
   *
   * @param variableName name of the release variable
   */
  @NoDoc
  VariableRefSpec variable(String variableName) {
    def varRef = new VariableRefSpec(variableName)
    variableReferences.add(varRef)
    varRef
  }

  @NoDoc
  @Override
  protected void postProcess() {
    super.postProcess()
    Task task = this.configurationItem as Task
    variableReferences.each { VariableRefSpec varRef ->
      if (!isGlobalOrFolderVariable(varRef.variableName)) {
        assertReleaseVariableExists(task, varRef)
        assertReleaseVariableOfCorrectType(task, varRef)
      }
      task.variableMapping[varRef.propertyName] = '${' + varRef.variableName + '}'
    }
  }

  @NoDoc
  protected static void assertReleaseVariableExists(Task task, VariableRefSpec variableRefSpec) {
    Release release = task.release
    boolean variableExist = release.variables.find { it.key == variableRefSpec.variableName }
    if (!variableExist) {
      CiProperty ciProperty = CiProperty.of(task, variableRefSpec.propertyName).get()
      def v = VariableFactory.createVariableByPropertyDescriptor(ciProperty.descriptor).get()
      v.setKey(variableRefSpec.variableName)
      release.addVariable(v)
      // throw new DslError("Variable '${variableRefSpec.variableName}' is not defined on a release '${release.title}'")
    }
  }

  @NoDoc
  protected static void assertReleaseVariableOfCorrectType(final Task task, VariableRefSpec variableRefSpec) {
    Release release = task.release
    Variable releaseVariable = release.variables.find { it.key == variableRefSpec.variableName }
    CiProperty ciProperty = CiProperty.of(task, variableRefSpec.propertyName).orElseThrow {
      throw new DslError("Variable '${variableRefSpec.variableName}' refers to non-existant property ${variableRefSpec.propertyName}")
    }
    Type expectedVariableType = VariableFactory.createVariableByPropertyDescriptor(ciProperty.descriptor).get().getType()
    if (expectedVariableType != releaseVariable.getType()) {
      throw new DslError("Variable '${variableRefSpec.variableName}' of type '${releaseVariable.getType()}' does not match expected type '${expectedVariableType}'")
    }
  }
}
