package com.xebialabs.xlrelease.dsl.service;

import java.util.List;
import java.util.Optional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.CreateReleaseTask;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.FolderService;
import com.xebialabs.xlrelease.service.ReleaseService;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_TEMPLATE;
import static java.lang.Integer.MAX_VALUE;
import static org.springframework.util.StringUtils.hasText;

@Component
public class CreateReleaseTaskCiProcessor implements CiProcessor<CreateReleaseTask> {

    private ReleaseService releaseService;
    private FolderService folderService;
    private PermissionChecker permissionChecker;

    @Autowired
    public CreateReleaseTaskCiProcessor(final ReleaseService releaseService,
                                        final FolderService folderService,
                                        final PermissionChecker permissionChecker) {
        this.releaseService = releaseService;
        this.folderService = folderService;
        this.permissionChecker = permissionChecker;
    }

    @Override
    public Type getType() {
        return Type.valueOf(CreateReleaseTask.class);
    }

    @Override
    public void process(final DslProcessingContext processingContext, final CreateReleaseTask ci) {
        String templateIdOrPath = ci.getTemplateId();

        if (VariableHelper.containsOnlyVariable(templateIdOrPath)) {
            ci.setTemplateId(templateIdOrPath);
        } else if (hasText(templateIdOrPath)) {
            try {
                String resolved = resolveTemplateIdFromPath(templateIdOrPath);
                ci.setTemplateId(resolved);
            } catch (DslError.TemplateByTitleNotFound | DslError.UnableToFindTemplateTitleInPath e) {
                checkTemplateIdExists(templateIdOrPath, e);
            }
        }

        String folderIdOrPath = ci.getFolderId();

        if (VariableHelper.containsOnlyVariable(folderIdOrPath)) {
            ci.setFolderId(folderIdOrPath);
        } else if (hasText(folderIdOrPath)) {
            try {
                String folderId = folderService.findByPath(folderIdOrPath, 0).getId();
                ci.setFolderId(folderId);
            } catch (NotFoundException e) {
                if (Ids.isFolderId(folderIdOrPath)) {
                    if (!folderService.exists(folderIdOrPath)) {
                        throw new DslError.UnableToFindFolderById(folderIdOrPath);
                    }
                } else {
                    throw new DslError.UnableToFindFolderByPath(folderIdOrPath);
                }
            }
        }

    }

    private void checkTemplateIdExists(final String templateId, final DslError originalException) {
        if (Ids.isReleaseId(templateId)) {
            if (!releaseService.exists(templateId)) {
                throw new DslError.TemplateByIdNotFound(templateId);
            }
        } else {
            throw originalException;
        }
    }

    private String resolveTemplateIdFromPath(final String fullTemplatePath) {
        Optional<String> folderPath = PathUtils.folderPath(fullTemplatePath);
        String folderId;
        try {
            folderId = folderPath.map(path -> folderService.findByPath(path, 0).getId()).orElse(ROOT_FOLDER_ID);
        } catch (NotFoundException e) {
            throw new DslError.UnableToFindTemplateTitleInPath(folderPath.orElse(ROOT_FOLDER_ID));
        }
        String templateTitle = templateTitle(fullTemplatePath);
        List<Release> templates = releaseService.findTemplatesByTitle(folderId, templateTitle, 0, 2, MAX_VALUE);

        if (templates.isEmpty()) {
            throw new DslError.TemplateByTitleNotFound(templateTitle);
        }
        if (templates.size() > 1) {
            throw new DslError.MoreThanOnTemplateWithTitle(templateTitle);
        }
        templates = permissionChecker.filter(templates, VIEW_TEMPLATE);
        if (templates.isEmpty()) {
            throw new DslError.NoViewPermissionOnTemplateTitle(templateTitle);
        }
        return templates.get(0).getId();
    }

    private String templateTitle(final String fullTemplatePath) {
        return PathUtils.releasePath(fullTemplatePath).orElseThrow(() ->
                new DslError.UnableToFindTemplateTitleInPath(fullTemplatePath)
        );
    }
}
