package com.xebialabs.xlrelease.dsl


import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.TemplateLogo
import com.xebialabs.xlrelease.domain.status.PhaseStatus
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import groovy.transform.TypeChecked

import static com.xebialabs.xlrelease.dsl.ReleaseAdditionalProperties.FOLDER
import static com.xebialabs.xlrelease.dsl.ReleaseAdditionalProperties.TARGET_FOLDER
import static com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE


/**
 * DSL representation of a Release
 */
@TypeChecked
class ReleaseSpec extends PlanItemSpec<Release> {
  ReleaseSpec(String title = null) {
    super(Type.valueOf(Release))
    Release release = configurationItem
    release.setStatus(ReleaseStatus.TEMPLATE)
    release.setTitle(title ?: "Release")
  }

  /**
   * Owner of the created release. If the owner field is not set, the "Run automated tasks as user" will be used.
   */
  @Override
  void owner(final String owner) {
    super.owner(owner)
  }

  /**
   * Releases automatically abort when a task fails if this property is set to true.
   */
  void abortOnFailure(boolean abortOnFailure) {
    configurationItem.abortOnFailure = abortOnFailure
  }

  /**
   * Release will be archived if this property is set to true.
   */
  void archiveRelease(boolean archiveRelease) {
    configurationItem.archiveRelease = archiveRelease
  }

  /**
   * Allows the usage of passwords in all fields in the UI. The passwords are masked in the UI and the log output.
   */
  void allowPasswordsInAllFields(boolean allowPasswordsInAllFields) {
    configurationItem.allowPasswordsInAllFields = allowPasswordsInAllFields
  }

  /**
   * Disable all notifications for this release.
   */
  void disableNotifications(boolean disableNotifications) {
    configurationItem.disableNotifications = disableNotifications
  }

  /**
   * Creates multiple Phases for a Release.
   */
  void phases(@DelegatesTo(value = PhasesDelegate, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    Release release = configurationItem
    release.phases = PhasesDelegate.delegate(new PhasesDelegate(release), cl)
    release.phases.each { it.status = PhaseStatus.PLANNED }
  }

  /**
   * Creates Variables for a Release.
   */
  void variables(@DelegatesTo(value = VariablesDelegate, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    def variables = configurationItem.getVariables().collect()
    variables.addAll(VariablesDelegate.delegate(new VariablesDelegate(), cl))
    configurationItem.setVariables(variables)
  }

  /**
   * The credentials of this user are used to run automated scripts in this release.
   */
  void scriptUsername(String username) {
    configurationItem.scriptUsername = username
  }

  /**
   * The password of the user that lends his credentials to run the scripts.
   */
  void scriptUserPassword(String password) {
    configurationItem.scriptUserPassword = password
  }

  /**
   * The tags of the release. Tags can be used for grouping and querying.
   */
  void tags(String... tags) {
    configurationItem.setTags((tags as Set) as List)
  }

  /**
   * If true, the release will automatically start at the scheduledStartDate.
   */
  void autoStart(boolean autoStart) {
    configurationItem.autoStart = autoStart
  }

  /**
   * If true, the releases calendar link will be published.
   */
  void calendarPublished(boolean calendarPublished) {
    configurationItem.calendarPublished = calendarPublished
  }

  /**
   * Creates teams with members, roles and permissions.
   */
  void teams(@DelegatesTo(value = TeamsDelegate, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    Release release = configurationItem
    release.teams = TeamsDelegate.delegate(new TeamsDelegate(release), cl)
  }

  /**
   * Configures extensions.
   */
  void extensions(@DelegatesTo(value = ReleaseExtensionsDelegate, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    Release release = configurationItem
    release.extensions = ReleaseExtensionsDelegate.delegate(new ReleaseExtensionsDelegate(release), cl)
  }

  /**
   * Add attachments.
   */
  void attachments(@DelegatesTo(value = ReleaseAttachmentsDelegate, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    Release release = configurationItem
    release.attachments = ReleaseAttachmentsDelegate.delegate(new ReleaseAttachmentsDelegate(release), cl)
  }

  void logo(String fileUri, @DelegatesTo(value = TemplateLogoSpec, strategy = Closure.DELEGATE_ONLY) Closure cl) {
    Release release = configurationItem
    TemplateLogoSpec spec = new TemplateLogoSpec(fileUri)
    TemplateLogo templateLogo = TemplateLogoSpec.delegate(spec, cl)
    release.logo = templateLogo
  }

  /**
   * The applied risk profile.
   */
  void riskProfile(RiskProfile riskProfile) {
    Release release = configurationItem
    release.setProperty(RISK_PROFILE, riskProfile)
  }

  /**
   * The folder in which to create the release in.
   */
  void folder(String folder) {
    additionalProperties.put(FOLDER, folder)
  }

  void defaultTargetFolder(String folder) {
    additionalProperties.put(TARGET_FOLDER, folder)
  }
}
