package com.xebialabs.xlrelease.api.v1.impl;

import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirContext;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.xlrelease.api.utils.ResponseHelper;
import com.xebialabs.xlrelease.api.v1.DslApi;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.export.ReleasefileExporter;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.ReleaseService;

import static com.google.common.base.Preconditions.checkArgument;

@Controller
public class DslApiImpl implements DslApi {

    private PermissionChecker permissions;
    private ReleaseService releaseService;
    private ReleasefileExporter releasefileExporter;

    @Autowired
    public DslApiImpl(final PermissionChecker permissions,
                      final ReleaseService releaseService,
                      final ReleasefileExporter releasefileExporter) {
        this.permissions = permissions;
        this.releaseService = releaseService;
        this.releasefileExporter = releasefileExporter;
    }

    @Override
    public Response previewExportTemplateToXFile(String templateId, boolean asTemplate) {
        permissions.checkView(templateId);
        checkArgument(releaseService.isTemplate(templateId), "Only templates can be exported.");

        final Release template = releaseService.findById(templateId);

        return Response.ok(releasefileExporter.exportAsString(template, asTemplate)).build();
    }

    @Override
    public Response exportTemplateToXFile(String templateId, boolean asTemplate) {
        permissions.checkView(templateId);
        checkArgument(releaseService.isTemplate(templateId), "Only templates can be exported.");

        final Release template = releaseService.findById(templateId);

        StreamingOutput output = outputStream -> {
            WorkDirContext.initWorkdir(WorkDirFactory.EXPORT_WORKDIR_PREFIX);
            final WorkDir workDir = WorkDirContext.get();
            try {
                releasefileExporter.exportZippedWithAttachments(template, asTemplate, outputStream);
            } finally {
                workDir.delete();
                WorkDirContext.clear();
            }
        };

        return ResponseHelper.streamFile(template.getTitle() + ".zip", output, "application/zip");
    }

    @Override
    public String exportTemplate(Release template) {
        return exportTemplate(template, false);
    }

    @Override
    public String exportTemplate(Release template, boolean asTemplate) {
        return apiExportTemplate(template, asTemplate);
    }

    private String apiExportTemplate(Release template, boolean renderAsTemplate) {
        checkArgument(releaseService.isTemplate(template.getId()), "Only templates can be exported.");
        permissions.checkView(template.getId());

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        releasefileExporter.exportAsReleasefile(template, renderAsTemplate, baos);
        try {
            return baos.toString(StandardCharsets.UTF_8.toString());
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }
}
