package com.xebialabs.xlrelease.dsl.service.renderer

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.Release.DEFAULT_TARGET_FOLDER_ID_PROPERTY_NAME
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.service.{FolderService, ReleaseService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import scala.jdk.CollectionConverters._

@Component
class ReleaseRenderer @Autowired()(
                                    val releaseService: ReleaseService,
                                    val folderService: FolderService)
  extends BaseTypedCiRenderer[Release]("release") with PathRenderingSupport {

  override def ignoredProperties = List("title", "status", "scmConnectorConfig")

  override def propertyOrder: Map[String, Int] = Map[String, Int](
    "variables" -> 0,
    "title" -> 1,
    "description" -> 2,
    "author" -> 19,
    "logo" -> 20,
    "allowTargetFolderOverride" -> 21,
    "phases" -> 100,
    "attachments" -> 110,
    "releaseTriggers" -> 120,
    "teams" -> 130,
    "extensions" -> 140
  )

  override def dsl(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    val overriddenDslKeyword = if (rendererContext.shouldRenderAsTemplate) {
      "template"
    } else {
      dslKeyword
    }

    s"""|$overriddenDslKeyword${renderTitle(ci)} {
        |${renderFolderAndProperties(rendererContext, ci)}
        |}""".stripMargin
  }

  override protected def shouldRender(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Boolean = {
    pd.getName match {
      case "releaseTriggers" => rendererContext.shouldRenderAsTemplate
      case "riskProfile" => Option(pd.get(ci)).exists(!_.asInstanceOf[RiskProfile].isDefaultProfile)
      case _ => super.shouldRender(rendererContext, ci, pd)
    }
  }

  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getName match {
      case "attachments" => {
        val release = ci.asInstanceOf[Release]
        val taskAttachments = release.getAllTasks.asScala.flatMap(t => t.getAttachments.asScala)
        val releaseAttachments = release.getAttachments.asScala.filterNot(taskAttachments.contains)

        super.renderCiCollection(rendererContext, releaseAttachments.asJava, pd)
      }
      case DEFAULT_TARGET_FOLDER_ID_PROPERTY_NAME =>
        val defaultTargetFolderId = ci.getProperty[String](DEFAULT_TARGET_FOLDER_ID_PROPERTY_NAME)
        if (StringUtils.hasText(defaultTargetFolderId)) {
          Some(s"defaultTargetFolder('${renderFolder(defaultTargetFolderId).dropRight(1)}')")
        } else {
          None
        }
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }

  def renderFolderAndProperties(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    val releaseId = ci.getId

    if (Ids.isInFolder(releaseId)) {
      s"""|folder('${renderFolder(releaseId).dropRight(1)}')
          |${renderProperties(rendererContext, ci)}
          |""".stripMargin
    } else {
      renderProperties(rendererContext, ci)
    }
  }
}
