package com.xebialabs.xlrelease.dsl.service;

import java.util.*;
import java.util.stream.Collectors;
import org.joda.time.LocalDateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plumbing.serialization.PortableConfigurationReference;
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.deployit.repository.core.Directory;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.builder.TeamBuilder;
import com.xebialabs.xlrelease.domain.BaseConfiguration;
import com.xebialabs.xlrelease.domain.PythonScript;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.domain.events.CreatedFromDsl;
import com.xebialabs.xlrelease.domain.events.PermissionsUpdatedEvent;
import com.xebialabs.xlrelease.domain.events.ReleaseCreatedEvent;
import com.xebialabs.xlrelease.domain.folder.Folder;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.domain.variables.ValueProviderConfiguration;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.ReleaseRepository;
import com.xebialabs.xlrelease.repository.SecuredCis;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.SecuredCi;
import com.xebialabs.xlrelease.service.CiIdService;
import com.xebialabs.xlrelease.service.FolderService;
import com.xebialabs.xlrelease.service.ReleaseService;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.utils.CiHelper;

import scala.Option;

import static com.xebialabs.deployit.core.rest.resteasy.WorkDirTemplate.cleanOnFinally;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plumbing.export.UnresolvedReferencesConfigurationItemConverter.DECRYPTION_FAILED_VALUE;
import static com.xebialabs.xlrelease.domain.Team.RELEASE_ADMIN_TEAMNAME;
import static com.xebialabs.xlrelease.domain.Team.TEMPLATE_OWNER_TEAMNAME;
import static com.xebialabs.xlrelease.dsl.ReleaseAdditionalProperties.FOLDER;
import static com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID;
import static com.xebialabs.xlrelease.repository.Ids.SEPARATOR;
import static com.xebialabs.xlrelease.repository.Ids.getParentId;
import static com.xebialabs.xlrelease.repository.Ids.isInFolder;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.CREATE_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getReleasePermissions;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getTemplateOnlyPermissions;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getTemplatePermissions;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables;
import static java.util.stream.Collectors.toList;
import static org.joda.time.LocalDateTime.fromDateFields;
import static org.joda.time.LocalDateTime.now;
import static org.springframework.util.StringUtils.hasText;

@Service
public class DslService implements ApiService {
    private static final Logger logger = LoggerFactory.getLogger(DslService.class);

    private TeamService teamService;
    private WorkDirFactory workDirFactory;
    private CiIdService ciIdService;
    private FolderService folderService;
    private CiProcessor ciProcessor;
    private PasswordEncrypter passwordEncrypter;
    private PermissionChecker permissionChecker;
    private ReleaseRepository releaseRepository;
    private ReleaseService releaseService;
    private ReleaseActorService releaseActorService;
    private ConfigurationRepository configurationRepository;
    private XLReleaseEventBus eventBus;
    private SecuredCis securedCis;

    @Autowired
    public DslService(WorkDirFactory workdirFactory,
                      CiIdService ciIdService,
                      TeamService teamService,
                      FolderService folderService,
                      CiProcessor ciProcessor,
                      PasswordEncrypter passwordEncrypter,
                      PermissionChecker permissionChecker,
                      ReleaseRepository releaseRepository,
                      ReleaseService releaseService,
                      ReleaseActorService releaseActorService,
                      ConfigurationRepository configurationRepository,
                      SecuredCis securedCis,
                      XLReleaseEventBus eventBus) {
        this.ciIdService = ciIdService;
        this.workDirFactory = workdirFactory;
        this.teamService = teamService;
        this.folderService = folderService;
        this.ciProcessor = ciProcessor;
        this.passwordEncrypter = passwordEncrypter;
        this.permissionChecker = permissionChecker;
        this.releaseRepository = releaseRepository;
        this.releaseService = releaseService;
        this.releaseActorService = releaseActorService;
        this.configurationRepository = configurationRepository;
        this.eventBus = eventBus;
        this.securedCis = securedCis;
    }

    @Timed
    public Release createRelease(Release parentRelease, Release release, Map<String, Object> additionalProperties) {
        DslProcessingContext processingContext = new DslProcessingContext();

        processingContext.setParentRelease(parentRelease);

        String folderIdOrPath = (String) additionalProperties.get(FOLDER);
        ConfigurationItem container = getContainer(folderIdOrPath, parentRelease);

        String containerId = (container == null) ? null : container.getId();
        permissionChecker.checkIsAllowedToCreateReleaseInFolder(containerId);

        if (release.getOwner() == null && parentRelease != null) {
            release.setOwner(parentRelease.getScriptUsername());
        }

        if (parentRelease != null) {
            processReleaseTeamsAndPermissions(parentRelease, release);
            if (containerId != null) {
                // Reset Teams to inherit folder permissions
                SecuredCi source = securedCis.getEffectiveSecuredCi(parentRelease.getId());
                SecuredCi destination = securedCis.getEffectiveSecuredCi(containerId);
                if (!source.getSecurityUid().equals(destination.getSecurityUid()))
                    release.setTeams(new ArrayList<>());
            }
        }



        release.setStatus(ReleaseStatus.PLANNED);

        create(container, release, release.getTeams(), processingContext);

        release.updateRealFlagStatus();

        initReleaseCalendarLinkToken(release);

        // 1. do the same things as Releases#createFromTemplate(templateId, releaseMetadata, createdFromTrigger)
        // 2. take template initialization from TemplateApi#create
        LocalDateTime releaseStart = release.getScheduledStartDate() != null ? fromDateFields(release.getScheduledStartDate()) : now();
        releaseService.setDatesFromTemplate(release, release, releaseStart);

        encryptPasswords(release);

        releaseRepository.update(release);

        releaseActorService.activate(release.getId());

        eventBus.publish(new ReleaseCreatedEvent(release, new CreatedFromDsl()));
        eventBus.publish(new PermissionsUpdatedEvent(release.getTeams()));

        return release;
    }

    @Timed
    public Release createTemplate(Release parentRelease, Release template, Map<String, Object> additionalProperties) {
        DslProcessingContext processingContext = new DslProcessingContext();

        processingContext.setParentRelease(parentRelease);

        String folderIdOrPath = (String) additionalProperties.get(FOLDER);
        ConfigurationItem container = getContainer(folderIdOrPath, parentRelease);
        processingContext.setContainer(container);

        if (processingContext.getContainer() != null) {
            permissionChecker.check(EDIT_TEMPLATE, processingContext.getContainer().getId());
        } else {
            permissionChecker.check(CREATE_TEMPLATE);
        }


        if (template.getOwner() == null && parentRelease != null) {
            template.setOwner(parentRelease.getScriptUsername());
        }

        processTemplateTeamsAndPermissions(processingContext, template);

        create(container, template, template.getTeams(), processingContext);

        // 1. do the same things as Releases#createFromTemplate(templateId, releaseMetadata, createdFromTrigger)
        // 2. take template initialization from TemplateApi#create
        LocalDateTime releaseStart = template.getScheduledStartDate() != null ? fromDateFields(template.getScheduledStartDate()) : now();
        releaseService.setDatesFromTemplate(template, template, releaseStart);

        encryptPasswords(template);

        releaseRepository.update(template);

        releaseActorService.activate(template.getId());

        eventBus.publish(new ReleaseCreatedEvent(template, new CreatedFromDsl()));
        eventBus.publish(new PermissionsUpdatedEvent(template.getTeams()));

        return template;
    }

    @Timed
    public Release importTemplate(Release template, String containerId) {
        DslProcessingContext processingContext = new DslProcessingContext();
        processingContext.setImport(true);

        ConfigurationItem container = null;
        if (!Ids.isRoot(containerId)) {
            container = folderService.findById(containerId, 0);
        }
        processingContext.setContainer(container);

        create(container, template, template.getTeams(), processingContext);

        return template;
    }

    private void processReleaseTeamsAndPermissions(Release parent, Release release) {
        if (release.getTeams().isEmpty() && !isInFolder(parent.getId())) {
            Team defaultReleaseAdminTeam = TeamBuilder.newTeam()
                    .withTeamName(RELEASE_ADMIN_TEAMNAME)
                    .withPermissions(getReleasePermissions())
                    .withMembers(parent.getScriptUsername())
                    .build();
            release.addTeam(defaultReleaseAdminTeam);
        }
    }

    private void processTemplateTeamsAndPermissions(DslProcessingContext processingContext, Release template) {
        if (processingContext.getContainer() != null) {
            template.setTeams(Collections.emptyList());
        } else {
            if (template.getTeams().isEmpty()) {
                Team defaultTemplateOwnerTeam = TeamBuilder.newTeam()
                        .withTeamName(TEMPLATE_OWNER_TEAMNAME)
                        .withPermissions(getTemplateOnlyPermissions())
                        .withMembers(processingContext.getParentRelease().getScriptUsername())
                        .build();

                List<String> releaseAdminTemplatePermissions = new ArrayList<>(getReleasePermissions());
                releaseAdminTemplatePermissions.add(VIEW_TEMPLATE.getPermissionName());

                Team defaultReleaseAdminTeam = TeamBuilder.newTeam()
                        .withTeamName(RELEASE_ADMIN_TEAMNAME)
                        .withPermissions(releaseAdminTemplatePermissions)
                        .build();

                template.setTeams(Arrays.asList(defaultTemplateOwnerTeam, defaultReleaseAdminTeam));
            } else {
                template.getTeams().forEach(team -> team.getPermissions().retainAll(getTemplatePermissions()));
            }
        }
    }

    private void encryptPasswords(ConfigurationItem item) {
        logger.debug("Encrypting password properties of `{}`", item.getId());
        CiTreeVisitor.of(item).with(ci ->
                CiHelper.forFields(ci, PropertyDescriptor::isPassword, this::encrypt)
        );
    }

    private void encrypt(final ConfigurationItem ci, final PropertyDescriptor pd) {
        logger.debug("Encrypting password property `{}` of `{}`", pd.getFqn(), ci.getId());
        final String passwordValue = (String) pd.get(ci);
        if (passwordValue != null) {
            String encryptedValue = passwordEncrypter.ensureEncrypted(passwordValue);
            pd.set(ci, encryptedValue);
        }
    }

    private Directory getContainer(String id) {
        String releaseId = releaseIdFrom(id);
        final String parentId = getParentId(releaseId);
        // parent of a release is either root or folder
        if (Ids.isRoot(parentId)) {
            return null;
        } else {
            return folderService.findById(parentId, 0);
        }
    }

    private Directory getContainer(String folderIdOrPath, Release parentRelease) {
        if (hasText(folderIdOrPath)) {
            if (!folderService.exists(folderIdOrPath)) {
                Folder folder = folderService.findByPath(folderIdOrPath, 0);

                if (null == folder) {
                    throw new DslError("No folder found with title or id '%s'.", folderIdOrPath);
                }

                return folder;
            }

            return folderService.findById(folderIdOrPath, 0);
        }

        return getContainer(parentRelease.getId());
    }

    private void create(ConfigurationItem parentContainer, ConfigurationItem ci, List<Team> teams, final DslProcessingContext processingContext) {
        processingContext.setContainer(parentContainer);
        createCis(getNestedCisAndPopulateWithId(parentContainer, ci, processingContext), teams, processingContext);
    }

    private void createCis(List<ConfigurationItem> cis, List<Team> teams, DslProcessingContext processingContext) {
        final List<Variable> newVariables = buildVariablesIfNeeded(cis);
        cis.addAll(newVariables);
        createCisAttachmentsAndNoTeams(cis, processingContext);
        saveTeams(teams, processingContext);
    }

    private List<ConfigurationItem> getNestedCisAndPopulateWithId(ConfigurationItem parent, ConfigurationItem ci, final DslProcessingContext processingContext) {
        List<ConfigurationItem> listWithNested = new ArrayList<>();
        listWithNested.add(ci);

        createId(ci, parent);

        //noinspection unchecked
        ciProcessor.process(processingContext, ci);

        for (PropertyDescriptor property : ci.getType().getDescriptor().getPropertyDescriptors()) {
            PropertyKind kind = property.getKind();

            if (kind == SET_OF_CI || kind == LIST_OF_CI) {
                Collection<ConfigurationItem> children = getChildren(ci, property);
                listWithNested.addAll(getNestedCis(ci, children, processingContext));
            }

            if (kind == CI) {
                Object unresolvedCiRef = property.get(ci);
                if (unresolvedCiRef instanceof PortableConfigurationReference) {
                    PortableConfigurationReference reference = (PortableConfigurationReference) unresolvedCiRef;
                    Option<String> folderIdOption = Option.apply(processingContext.getContainer()).map(ConfigurationItem::getId);
                    BaseConfiguration resolvedConfigurationItem = reference.resolve(configurationRepository, ResolutionContext.apply(folderIdOption)).getOrElse(() -> {
                        throw new DslError("No CI found for the type '%s' and title '%s'", reference.ciType().toString(), reference.title());
                    });
                    property.set(ci, resolvedConfigurationItem);
                } else {
                    ConfigurationItem ciRef = (ConfigurationItem) unresolvedCiRef;
                    if (ciRef != null) {
                        if (property.isNested() || isChild(ciRef, ci) || property.getReferencedType().instanceOf(Type.valueOf(PythonScript.class))) {
                            listWithNested.addAll(getNestedCisAndPopulateWithId(ci, ciRef, processingContext));
                        } else {
                            property.set(ci, ciRef);
                        }
                    }
                }
            }
            encryptPassword(ci, property);
        }

        return listWithNested;
    }

    private void encryptPassword(final ConfigurationItem ci, final PropertyDescriptor property) {
        if ((property.getKind() == PropertyKind.STRING) && property.isPassword() && property.get(ci) != null) {
            try {
                passwordEncrypter.ensureEncrypted(property.get(ci).toString());
            } catch (IllegalStateException | DeployitException ex) {
                property.set(ci, DECRYPTION_FAILED_VALUE);
                logger.info("Failed to read password field: '{}' with value: '{}'", property, property.get(ci), ex);
            }
        }
    }

    private List<ConfigurationItem> getNestedCis(ConfigurationItem parent, Collection<? extends ConfigurationItem> cis, final DslProcessingContext processingContext) {
        List<ConfigurationItem> listWithNested = new ArrayList<>();

        for (ConfigurationItem ci : cis) {
            listWithNested.addAll(getNestedCisAndPopulateWithId(parent, ci, processingContext));
        }

        return listWithNested;
    }

    private Collection<ConfigurationItem> getChildren(ConfigurationItem parent, PropertyDescriptor property) {
        @SuppressWarnings("unchecked")
        Collection<ConfigurationItem> references = (Collection<ConfigurationItem>) property.get(parent);

        // Parent/child relationship is modeled on the parent
        if (property.isAsContainment()) {
            return references;
        }

        // Parent/child relationship is modeled on the child
        return references.stream()
                .filter(ci -> isChild(ci, parent))
                .collect(Collectors.toList());
    }

    private boolean isChild(ConfigurationItem ci, ConfigurationItem parent) {
        for (PropertyDescriptor property : ci.getType().getDescriptor().getPropertyDescriptors()) {
            PropertyKind kind = property.getKind();

            if (kind == CI && property.isAsContainment() && parent.equals(property.get(ci))) {
                return true;
            }
        }
        return false;
    }

    private void createId(ConfigurationItem ci, ConfigurationItem parent) {
        String parentId = null == parent ? ROOT_FOLDER_ID : parent.getId();

        String id;
        if (ci.getType().equals(Type.valueOf(Dashboard.class))) {
            Checks.checkNotNull(parent, "Dashboard must have parent");
            id = parentId + SEPARATOR + Dashboard.DASHBOARD_PREFIX;
        } else if (ci.getType().isSubTypeOf(Type.valueOf(ValueProviderConfiguration.class))) {
            Checks.checkNotNull(parent, "ValueProviderConfiguration must be stored under Variable");
            id = parentId + "/valueProvider";
        } else {
            id = ciIdService.getUniqueId(ci.getType(), parentId);
        }

        ci.setId(id);
    }

    private void createCisAttachmentsAndNoTeams(final List<ConfigurationItem> cis, final DslProcessingContext processingContext) {
        List<ConfigurationItem> cisWithoutTeams = cis.stream()
                .filter(ci -> !ci.getType().instanceOf(Type.valueOf(Team.class)))
                .collect(toList());
        WorkDir workDir = workDirFactory.newWorkDir(WorkDirFactory.EXTERNAL_WORKDIR_PREFIX);
        if (!processingContext.isImport()) {
            cleanOnFinally(workDir, workDir1 -> {
                cisWithoutTeams.forEach(ci -> {
                    if (ci instanceof Release) {
                        releaseRepository.create((Release) ci, new CreatedFromDsl());
                    }
                });
                return null;
            });
        }
    }

    private List<Variable> buildVariablesIfNeeded(List<ConfigurationItem> entities) {
        return entities.stream()
                .filter(entity -> entity.getType().equals(Type.valueOf(Release.class)))
                .flatMap(entity -> {
                    Release release = (Release) entity;
                    return scanAndBuildNewVariables(release, release, ciIdService).stream();
                }).collect(toList());
    }

    private void saveTeams(Collection<Team> entities, DslProcessingContext processingContext) {
        // Group eventual teams by releases or folders
        Map<String, List<Team>> containersToTeams = new HashMap<>();
        entities.forEach(team -> {
            String containerId = getParentId(team.getId());
            team.setId(null);
            containersToTeams
                    .computeIfAbsent(containerId, s -> new ArrayList<>())
                    .add(team);
        });

        // Save platform permissions using the teamService
        if (!processingContext.isImport()) {
            containersToTeams.forEach((containerId, containerTeams) -> teamService.saveTeamsToPlatform(containerId, containerTeams));
        }
    }

    private void initReleaseCalendarLinkToken(Release release) {
        long token = UUID.randomUUID().getMostSignificantBits();
        release.setCalendarLinkToken(Long.toString(token));
    }

    @Override
    public String serviceName() {
        return "dsl";
    }
}


