package com.xebialabs.xlrelease.dsl

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.Metadata
import com.xebialabs.deployit.plumbing.serialization.PortableConfigurationReference
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.dsl.service.DslError
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import groovy.transform.TypeChecked

import static com.xebialabs.xlrelease.domain.CustomScriptTask.UNKNOWN_TYPE

@TypeChecked
abstract class ConfigurationItemSpec<T extends ConfigurationItem> {

  private Type type
  private T ci
  private Map<String, Object> additionalProperties = [:]

  ConfigurationItemSpec(Type ciType) {
    type = ciType
  }

  /**
   * setting the type property on a closure will reset the ConfigurationItem
   * and hence all properties set via method attributes will be lost.
   *
   * @param type
   */
  @NoDoc
  void type(String type) {
    setType(Type.valueOf(type))
  }

  /**
   * setting the type property on a closure will reset the ConfigurationItem
   * and hence all properties set via method attributes will be lost.
   *
   * @param type
   */
  @NoDoc
  void type(Type type) {
    setType(type)
  }

  Type getType() {
    return type
  }

  Map<String, Object> getAdditionalProperties() {
    return additionalProperties
  }

  def propertyMissing(String name, Object value) {
    def ci = getConfigurationItem()
    def pd = getPropertyDescriptor(name)
    pd.set(ci, value)
  }

  def propertyMissing(String name) {
    def ci = getConfigurationItem()
    def pd = getPropertyDescriptor(name)
    return pd.get(ci)
  }

  def methodMissing(String name, args) {
    def ci = getConfigurationItem()
    def param = args as Object[]
    def pd = getPropertyDescriptor(name)
    if (null != args) {
      if (param[0] instanceof String) {
        switch (pd.kind) {
          case PropertyKind.LIST_OF_STRING:
            pd.set(ci, param.toList())
            break
          case PropertyKind.SET_OF_STRING:
            pd.set(ci, param.toList().toSet())
            break
          case PropertyKind.CI:
            Type type = pd.getReferencedType()
            if (type.instanceOf(Type.valueOf(Configuration)) || type.instanceOf(Type.valueOf(RiskProfile))) {
              String portableId = Metadata.ConfigurationItemRoot.CONFIGURATION.rootNodeName + "/ignored/" + param[0] as String
              pd.set(ci, PortableConfigurationReference.from(type, portableId).get())
            } else {
              throw new DslError("Property '%s' should be a CI of type '%s'", name, pd.getReferencedType())
            }
            break
          default:
            pd.set(ci, param[0])
        }
      } else if (param[0] instanceof VariableRefSpec) {
        VariableRefSpec varRef = param[0] as VariableRefSpec
        varRef.setPropertyName(name)
      } else {
        pd.set(ci, param[0])
      }
    } else {
      return pd.get(ci)
    }
  }

  void setType(Type type) {
    if (type == UNKNOWN_TYPE) {
      throw new DslError("Cannot create '$UNKNOWN_TYPE' task")
    }
    this.type = type
    if (null != ci) {
      ci = null
    }
  }

  T getConfigurationItem() {
    if (ci == null) {
      ci = (getType().descriptor.newInstance(null) as T)
    }
    return ci
  }

  PropertyDescriptor getPropertyDescriptor(String name) {
    def descriptor = getType().getDescriptor().getPropertyDescriptors().find { pd ->
      pd.name == name || (pd.aliases != null && pd.aliases.contains(name))
    }
    if (descriptor == null) throw new DslError("ConfigurationItem of type '%s' does not have property '%s'", type, name)
    return descriptor
  }

  @NoDoc
  protected void postProcess() {
    // nothing
  }

  static <U extends ConfigurationItemSpec<T>, T> T delegate(int delegationStrategy, U planItemSpec, Closure cl) {
    def code = cl.rehydrate(planItemSpec, planItemSpec, planItemSpec)
    cl.resolveStrategy = delegationStrategy
    code()
    planItemSpec.configurationItem
  }

  static <U extends ConfigurationItemSpec<T>, T> T delegate(U planItemSpec, Closure cl) {
    delegate(Closure.DELEGATE_ONLY, planItemSpec, cl)
  }

}