package com.xebialabs.xlrelease.dsl.service.renderer

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.dsl.resolver.DeliveryTaskPathResolver._
import com.xebialabs.xlrelease.dsl.resolver.DeliveryTaskToPathResolver
import com.xebialabs.xlrelease.dsl.service.renderer.DslRenderer.{BEGIN_SCRIPT_TAG, END_SCRIPT_TAG}
import com.xebialabs.xlrelease.repository.IdType.DOMAIN
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.service.{FolderService, ReleaseService}
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.variable.VariableHelper.containsOnlyVariable
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import scala.jdk.CollectionConverters._
import scala.reflect.runtime.universe._
import scala.util.{Failure, Success, Try}

@Component
class CommentRenderer extends BaseTypedCiRenderer[Comment]("comment")

@Component
class TaskRenderer extends BaseTaskRenderer[Task]("manual")

class ScriptTaskRenderer[T <: ResolvableScriptTask](dslKeyword: String)(implicit clazzTag: TypeTag[T]) extends BaseTaskRenderer[T](dslKeyword) {

  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getKind match {
      case STRING if pd.getName == "script" =>
        val scriptTask = ci.asInstanceOf[T]
        Some(
          s"""script (['''\\
             |$BEGIN_SCRIPT_TAG${chunk(escapeMultiLineString(scriptTask.getScript))}
             |'''])$END_SCRIPT_TAG""".stripMargin)
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }

  def chunk(s: String, maxSize: Int = 8 * 1024): String = {
    val lines = Predef.augmentString(s).linesIterator
    val chunked: List[String] = lines.foldLeft(List[String]()) {
      (acc: List[String], line: String) => {
        val r = acc match {
          case head :: tail =>
            val combined = List(head, line).mkString("\n")
            if (combined.length > maxSize) {
              line :: acc
            } else {
              combined :: tail
            }
          case Nil => List(line)
        }
        r
      }
    }
    chunked.reverse.mkString("\n''',\n'''")
  }

  private def escapeMultiLineString(input: String): String = {
    input
      .replace("\\", "\\\\")
      .replaceAll("([^'])'([^'])", "$1\'$2")
      .replace("'''", "\\'\\'\\'")
  }
}

@Component
class JythonScriptTaskRenderer extends ScriptTaskRenderer[ScriptTask]("script")

@Component
class GroovyScriptTaskRenderer extends ScriptTaskRenderer[GroovyScriptTask]("groovyScript")

@Component
class NotificationTaskRenderer extends BaseTaskRenderer[NotificationTask]("notification")

@Component
class ParallelGroupTaskRenderer extends BaseTaskRenderer[ParallelGroup]("parallelGroup") {

  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getKind match {
      case SET_OF_CI if pd.getName == "links" =>
        val parallelGroup = ci.asInstanceOf[ParallelGroup]
        val items = pd.get(ci).asInstanceOf[util.Set[Link]]
        val sortedLinks = items.asScala.toList.sortBy(link => parallelGroup.getTasks.indexOf(link.getSource)).asJava
        renderCiCollection(rendererContext, sortedLinks, pd)
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }

}

@Component
class LinkRenderer extends BaseTypedCiRenderer[Link]("link") {

  override def ignoredProperties: List[String] = List("parallelGroup")

  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getKind match {
      case CI if pd.getName == "source" || pd.getName == "target" =>
        val task = pd.get(ci).asInstanceOf[Task]
        Some(s"${formatName(pd)} ${string(task.getTitle)}")
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }

}

@Component
class FacetRenderer extends BaseTypedCiRenderer[Facet]("facet") {

  override def ignoredProperties: List[String] = List("targetId")

  override def dsl(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    s"""|$dslKeyword('${ci.getType}') {
        |${renderProperties(rendererContext, ci)}
        |}""".stripMargin
  }

}

@Component
class SequentialGroupTaskRenderer extends BaseTaskRenderer[SequentialGroup]("sequentialGroup")

@Component
class UserInputTaskRenderer extends BaseTaskRenderer[UserInputTask]("userInput") {

  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getKind match {
      case LIST_OF_CI if "variables" == pd.getName => Option(
        s"""variables {
           |${pd.get(ci).asInstanceOf[java.util.Collection[Variable]].asScala.map(v => s"variable '${v.getKey}'").mkString("\n")}
           |}""".stripMargin)
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }

}

@Component
class CreateReleaseTaskRenderer @Autowired()(val releaseService: ReleaseService, val folderService: FolderService)
  extends BaseTaskRenderer[CreateReleaseTask]("createRelease") with PathRenderingSupport {


  override protected def shouldRender(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Boolean = {
    pd.getName match {
      case "riskProfile" => Option(pd.get(ci)).exists(!_.asInstanceOf[RiskProfile].isDefaultProfile)
      case _ => super.shouldRender(rendererContext, ci, pd)
    }
  }

  //noinspection ScalaStyle
  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getName match {
      case "templateId" => {
        val templateId = pd.get(ci).asInstanceOf[String]
        try {
          if (isIdFromVariable(templateId)) {
            Option(s"template '${DOMAIN.convertToViewId(templateId)}'")
          } else {
            val templateTitle = releaseService.getTitle(templateId)
            Option(s"template '${if (Ids.isInFolder(templateId)) renderFolder(templateId) else ""}$templateTitle'")
          }
        } catch {
          case ex: NotFoundException => Option(s"""|// Error (template): ${ex.getMessage}""")
        }
      }
      case "folderId" => {
        val folderId = pd.get(ci).asInstanceOf[String]
        Try {
          if (isIdFromVariable(folderId)) {
            Option(s"folder '${DOMAIN.convertToViewId(folderId)}'")
          } else {
            Option(s"folder '${renderFolder(folderId).dropRight(1)}'")
          }
        } match {
          case Success(rendered) => rendered
          case Failure(ex) => Option(s"""|// Error (folder): ${ex.getMessage}""")
        }
      }
      case "templateVariables" =>
        // the variableMapping contains keys like templateVariables[0].value
        val firstPartToCutOf = "templateVariables[".length
        val secondPartToCutOf = "].value".length

        def getVariableMappingIndex(key: String): Option[Int] = {
          Try(key.drop(firstPartToCutOf).dropRight(secondPartToCutOf).toInt).toOption
        }

        val variables = ci.getProperty(pd.getName).asInstanceOf[util.List[Variable]]
        val originalVariableMapping = rendererContext.currentCiVariableMapping
        val newVariableMapping = originalVariableMapping.asScala.map {
          case (key, value) => getVariableMappingIndex(key).fold(key -> value)(idx => variables.get(idx).getKey -> value)
        }
        rendererContext.currentCiVariableMapping = newVariableMapping.asJava
        val result = renderCiCollection(rendererContext, variables, pd)
        rendererContext.currentCiVariableMapping = originalVariableMapping
        result
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }

  private def isIdFromVariable(id: String): Boolean = {
    val variableOrId = DOMAIN.convertToViewId(id)
    containsOnlyVariable(variableOrId)
  }
}

@Component
class CustomScriptTaskRenderer extends BaseTaskRenderer[CustomScriptTask]("custom") {

  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    if ("pythonScript" == pd.getName) {
      Option(renderCi(rendererContext, pd.get(ci).asInstanceOf[PythonScript]))
    } else {
      super.renderProperty(rendererContext, ci, pd)
    }
  }

}

@Component
class PythonScriptRenderer extends BaseTypedCiRenderer[PythonScript]("script") {

  override def renderProperties(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    s"""type '${ci.getType}'
       |${super.renderProperties(rendererContext, ci)}""".stripMargin
  }

  override def dsl(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    s"""|$dslKeyword {
        |${renderProperties(rendererContext, ci)}
        |}""".stripMargin
  }

  override def ignoredProperties: List[String] = List("customScriptTask") ++ super.ignoredProperties
}

@Component
class DeliveryPythonScriptRenderer @Autowired()(val deliveryTaskToPathResolver: DeliveryTaskToPathResolver)
  extends PythonScriptRenderer {

  override def getType: Type = Type.valueOf(deliveryTaskType)

  //noinspection ScalaStyle
  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    deliveryTaskToPathResolver.resolve(ci, pd.getName)
      .get(pd)
      .fold(super.renderProperty(rendererContext, ci, pd)) { change =>
        change.newValue match {
          case Success(path) => Option(s"${getName(pd)} '$path'")
          case Failure(ex) => Option(s"""|// Error (${pd.getName}): ${ex.getMessage}""")
        }
      }
  }

  private def getName(pd: PropertyDescriptor): String = {
    if (pd.getCategory == CATEGORY_INPUT) pd.getName.replace("Id", "") else pd.getName
  }

}

@Component
class GateTaskRenderer extends BaseTaskRenderer[GateTask]("gate")

@Component
class GateConditionRenderer extends BaseTypedCiRenderer[GateCondition]("condition") {

  override def dsl(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    s"condition(${string(ci.getProperty("title"))})"
  }
}

@Component
class GateDependencyRenderer @Autowired()(
                                           val releaseService: ReleaseService,
                                           val folderService: FolderService)
  extends BaseTypedCiRenderer[Dependency]("dependency") with PathRenderingSupport {

  override def dsl(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    val bci = ci.asInstanceOf[Dependency]
    s"""|dependency {
        |${if (bci.getTarget != null) renderTarget(bci.getTarget[PlanItem]) else renderTargetId(bci.getTargetId)}
        |}""".stripMargin
  }

  private def renderTarget(targetItem: PlanItem): String = targetItem match {
    case release: Release =>
      s"""target {
         |${renderRelease(release)}
         |}""".stripMargin
    case phase: Phase =>
      s"""target {
         |${renderRelease(phase.getRelease)}
         |phase ${string(phase.getTitle)}
         |}""".stripMargin
    case task: Task =>
      s"""target {
         |${renderRelease(task.getRelease)}
         |phase ${string(task.getPhase.getTitle)}
         |task ${string(task.getTitle)}
         |}""".stripMargin
  }

  private def renderTargetId(targetId: String): String = {
    s"variable ${string(VariableHelper.withoutVariableSyntax(targetId))}"
  }

  private def renderRelease(release: Release): String = {
    val releaseId = release.getId
    val releaseTitle = release.getTitle
    s"release '${if (Ids.isInFolder(releaseId)) renderFolder(releaseId) else ""}$releaseTitle'"
  }
}

class BaseTaskRenderer[T <: Task](dslKeyword: String)(implicit clazzTag: TypeTag[T]) extends BaseTypedCiRenderer[T](dslKeyword)(clazzTag) {

  override def ignoredProperties = List("title", "status", "container", "variableMapping")

  override def propertyOrder: Map[String, Int] = Map[String, Int](
    "title" -> 1,
    "description" -> 2
  )

  override def renderProperties(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    val task = ci.asInstanceOf[Task]
    rendererContext.currentCiVariableMapping.putAll(task.getVariableMapping)
    val renderedDsl = super.renderProperties(rendererContext, ci)
    rendererContext.currentCiVariableMapping.clear()
    renderedDsl
  }
}
