package com.xebialabs.xlrelease.export

import com.google.common.base.Strings
import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.deployit.repository.WorkDirContext
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xlrelease.`export`.ReleasefileImporter.releaseFileImportScriptContext
import com.xebialabs.xlrelease.domain.Attachment.EXPORT_DIRECTORY
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.export.ReleasefileExporter.RELEASEFILE_FILENAME
import com.xebialabs.xlrelease.script.builder.ScriptContextBuilder
import com.xebialabs.xlrelease.script.groovy.GroovyScriptService
import com.xebialabs.xlrelease.script.{XlrScript, XlrScriptContext}
import com.xebialabs.xlrelease.service.{ReleaseService, TaskAccessService}
import org.apache.commons.io.{FilenameUtils, IOUtils}
import org.springframework.stereotype.Component

import java.io.InputStream
import java.nio.charset.StandardCharsets
import java.util
import java.util.Optional
import java.util.zip.{ZipEntry, ZipInputStream}
import javax.script.ScriptContext
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Using

@Component
class ReleasefileImporter(taskAccessService: TaskAccessService,
                          releaseService: ReleaseService,
                          templateProcessors: Optional[util.List[TemplateImportProcessor]],
                          groovyScriptService: GroovyScriptService)
  extends BaseTemplateImporter[ReleasefileTemplateContent](taskAccessService, releaseService, templateProcessors) {

  override def supports[U <: TemplateImportContext](importContext: U): Boolean = importContext match {
    case TemplateImportContext(_, isJson, isDefaultTemplate, isDsl, _) => isDsl && !isDefaultTemplate && !isJson
    case _ => false
  }

  override protected def doReadTemplateContent(inputStream: InputStream, importContext: TemplateImportContext): ReleasefileTemplateContent = {
    val zipEntries = toZipEntryMap(inputStream) { e =>
      val entryName = e.getName
      entryName.startsWith(EXPORT_DIRECTORY) || entryName == RELEASEFILE_FILENAME
    }

    val dslContent: String = zipEntries.get(RELEASEFILE_FILENAME).map(OverthereUtils.read(_, StandardCharsets.UTF_8.name())).orNull
    val localFiles = zipEntries - RELEASEFILE_FILENAME

    checkArgument(dslContent != null, s"Missing file in template export : $RELEASEFILE_FILENAME. Check if file exists in the root of the archive.")

    ReleasefileTemplateContent(dslContent, localFiles.asJava)
  }

  def toZipEntryMap(inputStream: InputStream)(predicate: ZipEntry => Boolean): Map[String, LocalFile] = {
    val acc: mutable.Map[String, LocalFile] = new mutable.HashMap[String, LocalFile]()
    val workDir = WorkDirContext.get
    Using.resource(new ZipInputStream(inputStream)) { stream =>
      for (entry <- LazyList.continually(stream.getNextEntry).takeWhile(_ != null)) {
        if (predicate(entry)) {
          // check that file names are correct
          val fileName = extractFileName(entry.getName)
          if (!Strings.isNullOrEmpty(fileName)) {
            val localFile = workDir.newFile(fileName)
            Using.resource(localFile.getOutputStream) { out =>
              IOUtils.copy(stream, out, TemplateImporter.BUFFER_SIZE)
            }
            acc += (entry.getName -> localFile)
          }
        }
        stream.closeEntry()
      }
    }
    acc.toMap
  }

  private def extractFileName(entryName: String): String = {
    FilenameUtils.getName(entryName)
  }

  override protected def doUpgradeTemplateContent(templateContent: ReleasefileTemplateContent): util.List[String] = {
    // do not upgrade dsl files
    new util.ArrayList[String]()
  }

  override private[export] def doConvertToTemplate(templateContent: ReleasefileTemplateContent, destinationFolderId: String): Release = {
    // execute dslContent and create in-memory Release object
    val scriptContext = releaseFileImportScriptContext(templateContent, destinationFolderId)
    groovyScriptService.executeScript(scriptContext).asInstanceOf[Release]
  }
}

case class ReleasefileTemplateContent(dslContent: String, localFiles: util.Map[String, LocalFile]) extends TemplateContent

object ReleasefileImporter {

  def releaseFileImportScriptContext(templateContent: ReleasefileTemplateContent, destinationFolderId: String): XlrScriptContext = {
    new ReleaseFileImportScriptContextBuilder(templateContent, destinationFolderId).build()
  }

  private class ReleaseFileImportScriptContextBuilder(templateContent: ReleasefileTemplateContent, destinationFolderId: String) extends ScriptContextBuilder {
    withScriptApi().withOpenApi().withGroovyImportDsl()

    override protected def doBuild(context: XlrScriptContext): Unit = {
      context.setAttribute("_importDestinationFolderId", destinationFolderId, ScriptContext.ENGINE_SCOPE)
      context.addScript(XlrScript.byContent(name = "<release_file_import>", content = templateContent.dslContent, wrap = false, checkPermissions = true))
    }

  }

}
