package com.xebialabs.xlrelease.dsl.resolver

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.api.v1.forms.DeliveryFilters
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.security.DeliveryPermissionChecker
import com.xebialabs.xlrelease.domain.delivery.DeliveryStatus
import com.xebialabs.xlrelease.dsl.resolver.DeliveryTaskPathResolver.{deliveryIdField, folderIdField, patternIdField, stageIdField}
import com.xebialabs.xlrelease.dsl.service.{DslError, PathUtils}
import com.xebialabs.xlrelease.repository.Ids.{getParentId, isDeliveryId, isStageId}
import com.xebialabs.xlrelease.service.FolderService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.mutable
import scala.compat.java8.OptionConverters._
import scala.util.Try

@Component
class DeliveryTaskFromPathResolver @Autowired()(val deliveryRepository: DeliveryRepository,
                                                val folderService: FolderService,
                                                val permissionChecker: DeliveryPermissionChecker)

  extends DeliveryTaskPathResolver {

  private val pathResolvers: Seq[(String, Resolver)] = Seq(
    (deliveryIdField, deliveryPathResolver),
    (patternIdField, patternPathResolver),
    (stageIdField, stagePathResolver),
    (folderIdField, folderPathResolver)
  )

  override def resolve(ci: ConfigurationItem): Map[PropertyDescriptor, PathResolution] = resolveCi(ci, pathResolvers)

  private def deliveryPathResolver(ci: ConfigurationItem, property: String, value: String, resolutions: mutable.Map[String, PathResolution]): String = {
    pathResolver(property, value, DeliveryStatus.values().filter(_ != DeliveryStatus.TEMPLATE))
  }

  private def patternPathResolver(ci: ConfigurationItem, property: String, value: String, resolutions: mutable.Map[String, PathResolution]): String = {
    pathResolver(property, value, Array(DeliveryStatus.TEMPLATE))
  }

  private def pathResolver(property: String, value: String, deliveryStatuses: Array[DeliveryStatus]): String = {
    val resolvedId = if (isDeliveryId(value)) value else resolveIdFromPath(value, deliveryStatuses)
    checkIdExists(property, resolvedId)
    resolvedId
  }

  private def stagePathResolver(ci: ConfigurationItem, property: String, value: String, resolutions: mutable.Map[String, PathResolution]): String =
    if (isStageId(value)) {
      resolveStageIdFromIdOrPath(getParentId(value), value)
    } else {
      val deliveryId: String = resolutions.get(deliveryIdField)
        .filter(isNewValueDeliveryId)
        .orElse(resolutions.get(patternIdField))
        .filter(isNewValueDeliveryId)
        .map(_.newValue.get)
        .getOrElse(throw new DslError(s"Missing either delivery or pattern to resolve stage name '$value'"))
      resolveStageIdFromIdOrPath(deliveryId, value)
    }

  private def folderPathResolver(ci: ConfigurationItem, property: String, value: String, resolutions: mutable.Map[String, PathResolution]): String = {
    resolveFolderIdFromPathOrId(value)
  }

  private def isNewValueDeliveryId(change: PathResolution): Boolean =
    change.newValue.isSuccess && change.newValue.get != null && isDeliveryId(change.newValue.get)

  private def resolveIdFromPath(deliveryPath: String, deliveryStatuses: Array[DeliveryStatus]): String = {
    val folderPath = PathUtils.folderPath(deliveryPath).asScala
    val folderId = try {
      folderPath.map((path: String) => folderService.findByPath(path, 0).getId).orNull
    } catch {
      case _: NotFoundException =>
        throw new DslError(s"No folder found with path '${folderPath.getOrElse("")}'.")
    }

    val deliveryTitle: String = PathUtils.releasePath(deliveryPath).asScala.getOrElse(throw new DslError(s"Unable to find delivery title in '$deliveryPath'."))

    val deliveryIds = deliveryRepository.searchIds(new DeliveryFilters()
      .withStatuses(deliveryStatuses: _*)
      .withTitle(deliveryTitle)
      .withStrictTitleMatch(true)
      .withFolderId(folderId)
    ).toList

    deliveryIds match {
      case Nil => throw new DslError(s"No ${if (deliveryStatuses.contains(DeliveryStatus.TEMPLATE)) "pattern" else "delivery"} found with title '$deliveryTitle'.")
      case head :: Nil =>
        permissionChecker.checkView(head)
        head
      case _ => throw new DslError(s"More then one ${if (deliveryStatuses.contains(DeliveryStatus.TEMPLATE)) "pattern" else "delivery"} found with title '$deliveryTitle'.")
    }
  }

  private def resolveStageIdFromIdOrPath(deliveryId: String, idOrPath: String): String = {
    try {
      val delivery = deliveryRepository.read(deliveryId)
      delivery.getStageByIdOrTitle(idOrPath).getId
    } catch {
      case _: NotFoundException =>
        throw new DslError(s"No stage found with id or title '$idOrPath'.")
    }
  }

  private def resolveFolderIdFromPathOrId(pathOrId: String): String = {
    Try(folderService.findByPath(pathOrId, 0))
      .orElse(Try(folderService.findById(pathOrId)))
      .map(_.getId)
      .getOrElse(throw new DslError(s"No folder found with path or id '${pathOrId}'."))
  }

  private def checkIdExists(property: String, value: String): Unit = {
    try {
      val delivery = deliveryRepository.read(value)
      if (property == deliveryIdField && delivery.getStatus == DeliveryStatus.TEMPLATE) {
        throw new DslError(s"The referenced id '$value' is a pattern where delivery is expected.")
      } else if (property == patternIdField && delivery.getStatus != DeliveryStatus.TEMPLATE) {
        throw new DslError(s"The referenced id '$value' is a delivery where pattern is expected.")
      }
    } catch {
      case _: NotFoundException =>
        throw new DslError(s"No delivery or pattern found with id '$value'.")
    }
  }
}
