package com.xebialabs.xlrelease.api.internal;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import org.jboss.resteasy.annotations.cache.NoCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.api.InternalSecurityProxy;
import com.xebialabs.deployit.core.api.dto.RolePermissions;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.engine.api.security.RolePrincipals;
import com.xebialabs.deployit.engine.api.security.User;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.UserService;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.xlrelease.actors.ManagedActorRegistry$;
import com.xebialabs.xlrelease.actors.utils.ReleaseActorLifecycleUtils;
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters;
import com.xebialabs.xlrelease.configuration.TaskAccess;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.calendar.Blackout;
import com.xebialabs.xlrelease.domain.calendar.CalendarEntry;
import com.xebialabs.xlrelease.domain.calendar.SpecialDay;
import com.xebialabs.xlrelease.domain.events.CreatedWithoutTemplate;
import com.xebialabs.xlrelease.domain.events.Imported;
import com.xebialabs.xlrelease.domain.events.ReleaseCreatedEvent;
import com.xebialabs.xlrelease.domain.events.ReleaseDeletedEvent;
import com.xebialabs.xlrelease.domain.facet.Facet;
import com.xebialabs.xlrelease.domain.folder.Folder;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.repository.*;
import com.xebialabs.xlrelease.repository.sql.persistence.Schema;
import com.xebialabs.xlrelease.search.ReleaseSearchResult;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.SessionService;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.*;
import com.xebialabs.xlrelease.triggers.actors.TriggerLifecycle;
import com.xebialabs.xlrelease.triggers.service.TriggerService;
import com.xebialabs.xlrelease.views.FixturesUser;
import com.xebialabs.xlrelease.views.RolePermissionsView;
import com.xebialabs.xlrelease.views.RolePrincipalsView;

import scala.util.Random;

import static com.xebialabs.deployit.core.rest.resteasy.WorkDirTemplate.cleanOnFinally;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static com.xebialabs.xlrelease.db.ArchivedReleases.REPORT_RELEASES_ID_COLUMN;
import static com.xebialabs.xlrelease.db.ArchivedReleases.REPORT_RELEASES_TABLE_NAME;
import static com.xebialabs.xlrelease.db.ArchivedReleases.shortenId;
import static com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE;
import static com.xebialabs.xlrelease.utils.CiHelper.getNestedCis;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables;
import static jakarta.ws.rs.core.Response.Status.ACCEPTED;
import static jakarta.ws.rs.core.Response.ok;
import static jakarta.ws.rs.core.Response.status;
import static java.lang.String.format;
import static java.nio.file.StandardOpenOption.CREATE_NEW;
import static java.nio.file.StandardOpenOption.WRITE;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static org.springframework.util.FileCopyUtils.copy;

/**
 * Used in the testing environment to set the database to a known state.
 */
@Path("/fixtures")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
@NoCache
public class FixturesResource {

    private static final Logger logger = LoggerFactory.getLogger(FixturesResource.class);
    private final RetryTemplate retryTemplate;
    private final RetryTemplate releaseSearchRetryTemplate;

    private PermissionChecker permissions;
    private InternalSecurityProxy internalSecurityProxy;
    private UserService userService;
    private UserProfileService userProfileService;
    private SessionService sessionService;
    private ArchivingService archivingService;
    private ArchivingScheduleService archivingScheduleService;
    private WorkDirFactory workDirFactory;
    private VariableService variableService;
    private ReleaseActorLifecycleUtils releaseActorLifecycleUtils;
    private TeamService teamService;
    private ConfigurationService configurationService;
    private XLReleaseEventBus eventBus;
    private RoleService roleService;
    private ActivityLogRepository activityLogRepository;
    private FolderRepository folderRepository;
    private FolderVariableRepository folderVariableRepository;
    private ReleaseRepository releaseRepository;
    private CiIdService ciIdService;
    private TaskRepository taskRepository;
    private ConfigurationRepository configurationRepository;
    private CalendarEntryRepository calendarEntryRepository;
    private CalendarResource calendarResource;
    private FacetService facetService;
    private JdbcTemplate jdbcTemplate;
    private JdbcTemplate reportingJdbcTemplate;
    private TriggerRepository triggerRepository;
    private TriggerLifecycle<Trigger> triggerLifecycle;
    private TriggerService triggerService;
    private UserTokenService userTokenService;
    private ReleaseSearchService releaseSearchService;

    @Autowired
    public FixturesResource(PermissionChecker permissions,
                            InternalSecurityProxy internalSecurityProxy, UserService userService, UserProfileService userProfileService, SessionService sessionService,
                            ArchivingService archivingService, ArchivingScheduleService archivingScheduleService,
                            WorkDirFactory workDirFactory, VariableService variableService, ReleaseActorLifecycleUtils releaseActorLifecycleUtils,
                            TeamService teamService, ConfigurationService configurationService,
                            XLReleaseEventBus eventBus, RoleService roleService, ActivityLogRepository activityLogRepository,
                            FolderRepository folderRepository, FolderVariableRepository folderVariableRepository,
                            ReleaseRepository releaseRepository, CiIdService ciIdService,
                            TaskRepository taskRepository, ConfigurationRepository configurationRepository,
                            CalendarEntryRepository calendarEntryRepository, CalendarResource calendarResource,
                            FacetService facetService,
                            @Qualifier("xlrRepositoryJdbcTemplate") JdbcTemplate jdbcTemplate,
                            @Qualifier("reportingJdbcTemplate") JdbcTemplate reportingJdbcTemplate,
                            TriggerRepository triggerRepository,
                            TriggerLifecycle<Trigger> triggerLifecycle,
                            TriggerService triggerService,
                            UserTokenService userTokenService,
                            ReleaseSearchService releaseSearchService,
                            @Value("5")
                            Integer retryAttempts
    ) {
        this.permissions = permissions;
        this.internalSecurityProxy = internalSecurityProxy;
        this.userService = userService;
        this.userProfileService = userProfileService;
        this.sessionService = sessionService;
        this.archivingService = archivingService;
        this.archivingScheduleService = archivingScheduleService;
        this.workDirFactory = workDirFactory;
        this.variableService = variableService;
        this.releaseActorLifecycleUtils = releaseActorLifecycleUtils;
        this.teamService = teamService;
        this.configurationService = configurationService;
        this.eventBus = eventBus;
        this.roleService = roleService;
        this.activityLogRepository = activityLogRepository;
        this.folderRepository = folderRepository;
        this.folderVariableRepository = folderVariableRepository;
        this.releaseRepository = releaseRepository;
        this.ciIdService = ciIdService;
        this.taskRepository = taskRepository;
        this.configurationRepository = configurationRepository;
        this.calendarEntryRepository = calendarEntryRepository;
        this.calendarResource = calendarResource;
        this.facetService = facetService;
        this.jdbcTemplate = jdbcTemplate;
        this.reportingJdbcTemplate = reportingJdbcTemplate;
        this.triggerRepository = triggerRepository;
        this.triggerLifecycle = triggerLifecycle;
        this.triggerService = triggerService;
        this.userTokenService = userTokenService;
        this.releaseSearchService = releaseSearchService;
        this.retryTemplate = RetryTemplate.builder()
                .maxAttempts(retryAttempts)
                .exponentialBackoff(300, 2, 5000)
                .retryOn(RuntimeException.class)
                .build();
        this.releaseSearchRetryTemplate = RetryTemplate.builder()
                .maxAttempts(2)
                .fixedBackoff(1000)
                .build();
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @POST
    @Deprecated // use createRelease instead
    public void createEntities(List<ConfigurationItem> entities) {
        this.permissions.check(ADMIN);
        logger.debug(format("Creating entities: %s", entities));

        List<String> releaseIds = entities.stream()
                .filter(ci -> ci.getType().equals(Type.valueOf(Release.class)))
                .map(ConfigurationItem::getId)
                .collect(toList());

        createOrUpdateCis(entities);

        releaseIds.forEach(id -> {
            Release release = releaseRepository.findById(id);
            eventBus.publish(new ReleaseCreatedEvent(release, new Imported()));
        });
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @POST
    @Path("tree")
    @Deprecated // use createRelease instead
    public void createCiTree(ConfigurationItem ci) {
        this.permissions.check(ADMIN);
        logger.debug(format("Creating entity: %s", ci));
        createOrUpdateCis(getNestedCis(ci));
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @POST
    @Path("releases")
    public void createReleases(List<Release> releases) {
        releases.forEach(this::createRelease);
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @POST
    @Path("release")
    public void createRelease(Release release) {
        this.permissions.check(ADMIN);
        logger.debug(format("Creating release: %s", release));

        assignDefaultRiskProfileIfMissing(release);

        WorkDir workDir = workDirFactory.newWorkDir(WorkDirFactory.EXTERNAL_WORKDIR_PREFIX);
        cleanOnFinally(workDir, wd -> {
            List<Attachment> attachments = release.getAttachments();
            try {
                enrichAttachments(attachments, wd);

                scanAndBuildNewVariables(release, release, ciIdService);
                releaseRepository.create(release, new CreatedWithoutTemplate());
                eventBus.publish(new ReleaseCreatedEvent(release, null));
                return null;
            } finally {
                attachments.forEach(a -> {
                    a.getFile().delete();
                });
            }
        });

        release.getTeams().forEach(team -> team.setId(null));

        teamService.saveTeamsToPlatform(release);
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @POST
    @Path("activityLogs/{releaseId:.*}")
    public void createActivityLogs(@PathParam("releaseId") String releaseId, List<ActivityLogEntry> logs) {
        this.permissions.check(ADMIN);
        logger.debug(format("Creating activity logs: %s", logs));

        logs.forEach(log -> {
            if (log.hasUsername()) {
                activityLogRepository.log(releaseId, log, log.getUsername());
            } else {
                activityLogRepository.log(releaseId, log);
            }
        });
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @POST
    @Path("preArchived")
    public void createPreArchivedRelease(Release release) {
        this.permissions.check(ADMIN);
        createRelease(release);
        logger.debug(format("Pre-archiving release: %s", release));
        archivingService.preArchiveRelease(release);
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @POST
    @Path("archived")
    public void createArchivedRelease(Release release) {
        createPreArchivedRelease(release);
        logger.debug(format("Archiving release: %s", release));
        archiveRelease(release.getId());
    }

    @GET
    @Path("preArchive/{releaseId:.*}")
    public Response preArchiveRelease(@PathParam("releaseId") String releaseId) {
        this.permissions.check(ADMIN);
        Release release = releaseRepository.findById(releaseId);

        archivingService.preArchiveRelease(release);

        return status(ACCEPTED).build();
    }

    @Path("archive/{releaseId:.*}")
    @GET
    public Response archiveRelease(@PathParam("releaseId") String releaseId) {
        this.permissions.check(ADMIN);

        // make sure actors are stopped
        terminateRelease(releaseId);
        // releaseActorLifecycleUtils.terminateReleaseActorAndAwait(releaseId, FiniteDuration.apply(5, TimeUnit.SECONDS));
        archivingService.archiveRelease(releaseId);

        return status(ACCEPTED).build();
    }

    @POST
    @Path("archiveAll")
    public Response triggerArchivingOfCompletedReleases() {
        this.permissions.check(ADMIN);
        logger.debug("Archiving all finished releases");

        archivingScheduleService.processExpiredReleases(0);

        return ok().build();
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @PUT
    public void updateEntities(List<ConfigurationItem> entities) {
        this.permissions.check(ADMIN);
        logger.debug(format("Updating entities: %s", entities));
        List<ConfigurationItem> releaseCis = entities.stream().filter(ci -> ci instanceof PlanItem).collect(toList());
        if (!releaseCis.isEmpty()) {
            throw new UnsupportedOperationException("You cannot update parts of release using PUT /fixtures: " + releaseCis);
        }

        createOrUpdateCis(entities);
    }

    @Path("task-access")
    @DELETE
    public void deleteTaskAccess() {
        this.permissions.check(ADMIN);
        logger.debug("Deleting all task accesses");

        deleteAll(TaskAccess.class);
    }

    @Path("{id:(Applications|Folder|Template|Release|Custom|Configuration).*}")
    @DELETE
    public void deleteApplicationEntity(@PathParam("id") String id) {
        this.permissions.check(ADMIN);
        logger.debug(format("Deleting application entity: [%s]", id));
        deleteCi(id);
    }

    @Path("/configurations")
    @POST
    public void createConfigurationItems(List<BaseConfiguration> baseConfigurations) {
        this.permissions.check(ADMIN);
        logger.debug("Creating configuration entities");
        createOrUpdateConfiguration(baseConfigurations);
    }

    @Path("/configuration/{id:.*}")
    @DELETE
    public void deleteConfigurationItem(@PathParam("id") String id) {
        this.permissions.check(ADMIN);
        logger.debug(format("Deleting configuration entity: [%s]", id));

        deleteCi("Configuration/" + id);
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @Path("/user")
    @POST
    public void createUser(FixturesUser fixturesUser) {
        this.permissions.check(ADMIN);
        logger.debug(format("Creating user: %s", fixturesUser.getUsername()));

        User user = fixturesUser.toUser();
        try {
            userService.read(user.getUsername());
            logger.debug(format("User %s already exists", user.getUsername()));
        } catch (NotFoundException e) {
            userService.create(user.getUsername(), user.getPassword());
        }
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @Path("/folders")
    @POST
    public void createFolders(List<Folder> folders) {
        this.permissions.check(ADMIN);
        folders.stream()
                .sorted(Comparator.comparing(Folder::getId))
                .filter(folder -> !folderRepository.exists(folder.getId()))
                .forEach(folder -> folderRepository.create(Ids.getParentId(folder.getId()), folder));
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @Path("/folders/variables")
    @POST
    public List<Variable> createFolderVariables(List<Variable> variables) {
        this.permissions.check(ADMIN);
        return variables.stream()
                .filter(variable -> !this.folderVariableRepository.exists(variable.getId()))
                .map(variable -> this.folderVariableRepository.create(variable))
                .collect(toList());
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @Path("/folders/variable/{variableId}")
    @DELETE
    public void deleteFolderVariable(@PathParam("variableId") String variableId) {
        this.permissions.check(ADMIN);
        if (folderVariableRepository.exists(variableId)) {
            folderVariableRepository.delete(variableId);
        }
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @Path("/teams")
    @POST
    public void createTeams(List<Team> teams) {
        this.permissions.check(ADMIN);
        teams.stream()
                .collect(groupingBy(team -> Ids.getParentId(team.getId())))
                .forEach((containerId, teamsList) -> this.teamService.saveTeamsToPlatform(containerId, teamsList));
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @Path("/shared")
    @POST
    public void createSharedConfigurations(List<Configuration> configurations) {
        this.permissions.check(ADMIN);
        configurations.forEach(configuration -> configurationRepository.create(configuration));
    }

    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @Path("/shared")
    @DELETE
    public void deleteSharedConfigurations(List<String> configurationIds) {
        this.permissions.check(ADMIN);
        configurationIds.forEach(configurationId -> configurationRepository.delete(configurationId));
    }

    @Path("/user/{login}")
    @DELETE
    public void deleteUser(@PathParam("login") String login) {
        this.permissions.check(ADMIN);
        logger.debug(format("Deleting user: %s", login));

        userService.delete(login);
        userProfileService.deleteByUsername(login);
        sessionService.disconnect(login);
    }

    @Path("/role/{roleName}")
    @DELETE
    public void deleteRole(@PathParam("roleName") String roleName) {
        this.permissions.check(ADMIN);
        logger.debug(format("Deleting role: %s", roleName));
        try {
            roleService.deleteByName(roleName);
        } catch (NotFoundException e) {
            logger.warn(e.getMessage(), e);
        }
    }

    @Path("/calendar")
    @DELETE
    public void deleteCalendar() {
        this.permissions.check(ADMIN);
        logger.debug("Deleting all special days and blackouts from calendar");

        Stream.concat(
                calendarEntryRepository.findAllByType(Type.valueOf(Blackout.class)).stream(),
                calendarEntryRepository.findAllByType(Type.valueOf(SpecialDay.class)).stream()
        ).forEach(entry -> calendarEntryRepository.delete(entry.getId()));
    }

    @Path("/cis")
    @DELETE
    public void deleteCIs(List<ConfigurationItem> cis) {
        this.permissions.check(ADMIN);
        logger.debug(format("Deleting CIs: %s", cis));

        for (ConfigurationItem ci : cis) {
            deleteCi(ci.getId());
        }
    }

    @Path("/expectContainingAttachments/{releaseId:.*}")
    @POST
    @Workdir(prefix = WorkDirFactory.ARTIFACT_WORKDIR_PREFIX)
    public Response expectContainingAttachments(@PathParam("releaseId") String releaseId, ExpectedAttachment expectedAttachment) throws IOException {
        this.permissions.check(ADMIN);

        Release release = releaseRepository.findById(releaseId);
        for (Attachment attachment : release.getAttachments()) {
            OverthereFile file = attachment.getFile();
            if (file.getName().equals(expectedAttachment.getName())) {
                return checkContent(expectedAttachment, file);
            }
        }

        throw new Error("Attachment not found");
    }

    private Response checkContent(ExpectedAttachment expectedAttachment, OverthereFile file) throws IOException {
        try (InputStream inputStream = file.getInputStream()) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

            copy(inputStream, outputStream);
            String content = outputStream.toString(StandardCharsets.UTF_8.name());

            if (content.equals(expectedAttachment.getExpectedContent())) {
                return ok().build();
            } else {
                throw new Error("Expected content to be " + expectedAttachment.getExpectedContent() + " content found : " + content);
            }
        }
    }

    @POST
    @Path("roles/permissions/{id:.*?}")
    public void writeRolePermissions(@PathParam("id") String id, List<RolePermissionsView> rolePermissionsViews) {
        this.permissions.check(ADMIN);
        logger.debug(format("Setting %d role permissions for [%s]", rolePermissionsViews.size(), id));

        List<RolePermissions> permissions = rolePermissionsViews.stream().map(RolePermissionsView::toRolePermissions).collect(toList());
        permissions.forEach(p -> teamService.generateIdIfNecessary(p.getRole()));
        internalSecurityProxy.writeRolePermissions(id, permissions);
    }

    @POST
    @Path("roles/principals")
    public void writeRolePrincipals(List<RolePrincipalsView> rolePrincipalsViews) {
        this.permissions.check(ADMIN);
        logger.debug(format("Setting %d role principals", rolePrincipalsViews.size()));

        List<RolePrincipals> rolePrincipals = rolePrincipalsViews.stream().map(RolePrincipalsView::toRolePrincipals).collect(toList());
        rolePrincipals.forEach(rp -> teamService.generateIdIfNecessary(rp.getRole()));
        internalSecurityProxy.writeRolePrincipals(rolePrincipals);
    }

    @POST
    @Path("expectCommentAdded/{taskId:.*?}")
    public boolean checkCommentAdded(@PathParam("taskId") String taskId, String textToSearch) {
        Task task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS());

        for (Comment comment : task.getComments()) {
            if (comment.getText().contains(textToSearch)) {
                return true;
            }
        }

        return false;
    }


    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    @POST
    @Path("facet")
    public Facet createFacet(Facet facet) {
        permissions.check(ADMIN);
        return facetService.create(facet);
    }

    @GET
    @Path("auth")
    public boolean checkAuthentication() {
        return SecurityContextHolder.getContext().getAuthentication().isAuthenticated();
    }

    private void deleteAll(Class<? extends ConfigurationItem> ciClass) {
        this.permissions.check(ADMIN);
        Type ciType = Type.valueOf(ciClass);
        if (ciType.isSubTypeOf(Type.valueOf(BaseConfiguration.class))) {
            configurationRepository.findAllByType(ciType).forEach(ci -> configurationRepository.delete(ci.getId()));
        } else {
            throw new IllegalArgumentException(format("Unable to delete CIs of type: '%s'", ciType));
        }
    }

    @Path("/globalVariables/")
    @DELETE
    public void deleteGlobalVariables() {
        this.permissions.check(ADMIN);
        logger.debug("Deleting all global variables");

        variableService.findGlobalVariablesOrEmpty().getVariables().forEach(v -> variableService.deleteGlobalVariable(v.getId()));
    }

    @Path("/trigger")
    @POST
    public void createTrigger(Trigger trigger) {
        // TODO replace here as well with triggerService and fix tests
        Trigger addedTrigger = triggerRepository.create(trigger);
        triggerLifecycle.enable(addedTrigger, true);
    }

    @Path("/trigger/{triggerId:.*}")
    @DELETE
    public void deleteTrigger(@PathParam("triggerId") String triggerId) {
        triggerService.disableTriggers(List.of(triggerId));
        ReleasesFilters filter = new ReleasesFilters();
        String triggerTag = Ids.getName(triggerId);
        filter.setTags(List.of(triggerTag));
        try {
            releaseSearchRetryTemplate.execute(ctx -> {
                ReleaseSearchResult result = releaseSearchService.search(filter, 0, Integer.MAX_VALUE);
                logger.info(format("Found %d releases from trigger %s", result.getReleases().size(), triggerTag));
                result.getReleases().forEach(r -> deleteCi(r.getId()));
                if (result.getReleases().isEmpty()) {
                    throw new RuntimeException("No releases found. Try once more to confirm all are deleted!");
                }
                return null;
            });
        } catch (RuntimeException e) {
            //nothing to do in fixtures
        }

        triggerService.deleteTrigger(triggerId);
    }

    @Path("/tokens/{username}")
    @DELETE
    public void deleteUserTokens(@PathParam("username") String username) {
        this.permissions.check(ADMIN);
        logger.debug(format("Deleting all personal access token for user[%s]", username));
        userTokenService.deleteAllUserToken(username);
    }

    @Path("/categories")
    @DELETE
    public void deleteAllCategories() {
        this.permissions.check(ADMIN);
        logger.debug("Deleting all categories");
        jdbcTemplate.update(format("DELETE FROM %s", Schema.CATEGORIES$.MODULE$.TABLE()));
    }

    private void assignDefaultRiskProfileIfMissing(final Release release) {
        if (release.hasProperty(RISK_PROFILE) && release.getProperty(RISK_PROFILE) == null && !release.isWorkflow()) {
            release.setProperty(RISK_PROFILE, configurationRepository.read("Configuration/riskProfiles/RiskProfileDefault"));
        }
    }

    private void createOrUpdateCis(final List<ConfigurationItem> entities) {
        createOrUpdateCalendar(takeAllByType(entities, CalendarEntry.class));
        createOrUpdateConfiguration(takeAllByType(entities, BaseConfiguration.class).stream().map(BaseConfiguration.class::cast).collect(toList()));
        createOrUpdateTeams(takeAllByType(entities, Team.class));
        createOrUpdateOther(entities);
    }

    private void createOrUpdateCalendar(final List<ConfigurationItem> cis) {
        cis.forEach(ci -> {
            if (ci instanceof SpecialDay) {
                calendarResource.setSpecialDay(ci.getId(), (SpecialDay) ci);
            } else {
                calendarEntryRepository.createOrUpdate((CalendarEntry) ci);
            }
        });
    }

    private void createOrUpdateConfiguration(final List<BaseConfiguration> cis) {
        cis.forEach(configurationService::createOrUpdate);
    }

    private void createOrUpdateOther(final List<ConfigurationItem> cis) {
        if (!cis.isEmpty()) {
            String ciList = cis.stream().map((ConfigurationItem ci) -> ci.getId() + ci.getType()).collect(joining(","));
            throw new IllegalArgumentException(format("Unable to delete entities '%s'", ciList));
        }
    }

    private List<ConfigurationItem> takeAllByType(final List<ConfigurationItem> cis, Class<? extends ConfigurationItem> ciClass) {
        List<ConfigurationItem> items = cis.stream()
                .filter(ci -> ci.getType().instanceOf(Type.valueOf(ciClass)))
                .collect(toList());
        cis.removeAll(items);
        return items;
    }

    private void enrichAttachments(List<Attachment> attachments, WorkDir workDir) {
        for (Attachment attachment : attachments) {
            logger.debug("Enriching attachment " + attachment.getId());
            // create a dummy file for testing
            LocalFile dummyFile = workDir.newFile(attachment.getName() + ".txt");

            Random random = new Random();
            List<String> lines = findSizeSpecInAttachmentName(attachment.getName())
                    .map(sizeInKb -> IntStream.range(0, sizeInKb)
                            .mapToObj(i -> random.nextString(1024 / 3))
                            .collect(toList())).orElse(Collections.singletonList("test"));

            java.nio.file.Path dummyFilePath = Paths.get(dummyFile.getFile().toURI());
            try {
                Files.write(dummyFilePath, lines, CREATE_NEW, WRITE);
                attachment.setFile(dummyFile);
            } catch (IOException e) {
                throw new RuntimeException("Could not create dummy content for attachment " + attachment.getId(), e);
            }
        }
    }

    private void createOrUpdateTeams(Collection<ConfigurationItem> entities) {
        // Group eventual teams by releases or folders
        Map<String, List<Team>> containersToTeams = new HashMap<>();
        entities.forEach(team -> {
            String containerId = Ids.getParentId(team.getId());
            team.setId(null);
            containersToTeams
                    .computeIfAbsent(containerId, s -> new ArrayList<>())
                    .add((Team) team);
        });

        // Save platform permissions using the teamService
        containersToTeams.forEach(teamService::saveTeamsToPlatform);
    }

    private void deleteCi(String id) {
        if (Ids.isReleaseId(id)) {
            terminateRelease(id);
            deleteFromArchive(id);
            if (releaseRepository.exists(id)) {
                deleteReleaseWithRetry(id);
                Release deletedRelease = Type.valueOf(Release.class).getDescriptor().newInstance(id);
                eventBus.publish(new ReleaseDeletedEvent(deletedRelease));
            }
        } else if (Ids.isCalendarId(id)) {
            if (calendarEntryRepository.exists(id)) {
                calendarEntryRepository.delete(id);
            }
        } else if (Ids.isConfigurationId(id)) {
            if (configurationRepository.exists(id)) {
                configurationRepository.delete(id);
            }
        } else {
            throw new IllegalArgumentException(format("Unable to delete CI with id: %s", id));
        }
    }

    private void deleteReleaseWithRetry(final String releaseId) {
        retryTemplate.execute(ctx -> {
            logger.trace("Attempted to delete {} for {} time", releaseId, ctx.getRetryCount());
            releaseRepository.delete(releaseId, false);
            return null;
        });
    }

    private Optional<Integer> findSizeSpecInAttachmentName(final String name) {
        Pattern pattern = Pattern.compile(".*_(\\d+)KB.*");
        Matcher matcher = pattern.matcher(name);
        if (matcher.find()) {
            return Optional.of(Integer.parseInt(matcher.group(1)));
        }
        return Optional.empty();
    }

    private int deleteFromArchive(String id) {
        return reportingJdbcTemplate.update(format("DELETE FROM %s WHERE %s IN (?, ?)", REPORT_RELEASES_TABLE_NAME(), REPORT_RELEASES_ID_COLUMN()), shortenId(id), "Applications/" + shortenId(id));
    }

    private static class ExpectedAttachment {
        String name;
        String expectedContent;

        public ExpectedAttachment() {
        }

        public String getName() {
            return name;
        }

        @SuppressWarnings("WeakerAccess")
        public String getExpectedContent() {
            return expectedContent;
        }
    }


    private void terminateRelease(String id) {
        var releaseId = Ids.getFolderlessId(id);
        String actorName = null;
        try {
            actorName = releaseActorLifecycleUtils.terminateReleaseActor(releaseId);
        } catch (Exception e) {
            logger.error(format("Could not terminate release actor %s within timeout", id), e);
        } finally {
            if (actorName != null) {
                ManagedActorRegistry$.MODULE$.remove(actorName);
            }
        }
    }
}
