package com.xebialabs.xlrelease.api.internal;

import java.util.List;
import java.util.Objects;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.jboss.resteasy.annotations.cache.NoCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.UserProfileService;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static javax.ws.rs.core.Response.Status.NOT_FOUND;
import static javax.ws.rs.core.Response.ok;
import static javax.ws.rs.core.Response.status;

@Path("/fixtures")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
@NoCache
public class FixturesUserProfileResource {

    private static final Logger logger = LoggerFactory.getLogger(FixturesUserProfileResource.class);

    private PermissionChecker permissions;
    private UserProfileService userProfileService;

    public FixturesUserProfileResource(PermissionChecker permissions, UserProfileService userProfileService) {
        this.permissions = permissions;
        this.userProfileService = userProfileService;
    }

    @Path("/userProfile")
    @GET
    public Response searchUserProfile(@QueryParam("name") String username, @QueryParam("email") String userEmail, @QueryParam("fullName") String fullName) {
        this.permissions.check(ADMIN);

        UserProfile profile = userProfileService.findByUsername(username);

        if (profile == null) return status(NOT_FOUND).build();

        boolean sameEmail = Objects.equals(profile.getEmail(), userEmail);
        boolean sameFullName = Objects.equals(profile.getFullName(), fullName);
        return sameEmail && sameFullName ? ok().build() : status(NOT_FOUND).build();
    }

    @Path("/userProfile")
    @POST
    public void createUserProfile(UserProfile userProfile) {
        this.permissions.check(ADMIN);
        logger.debug(String.format("Creating user profile: %s", userProfile.getName()));

        userProfileService.save(userProfile);
    }

    @Path("/userProfiles")
    @POST
    public void createUserProfiles(List<UserProfile> userProfiles) {
        logger.debug("Creating user profiles");
        userProfiles.forEach(this::createUserProfile);
    }

    @Path("/userProfile")
    @PUT
    public void updateUserProfile(UserProfile userProfile) {
        this.permissions.check(ADMIN);
        logger.debug(String.format("Updating user profile: %s", userProfile.getName()));

        userProfileService.save(userProfile);
    }

    @Path("/userProfile/{username:.*}")
    @DELETE
    public void deleteUserProfile(@PathParam("username") String username) {
        this.permissions.check(ADMIN);
        logger.debug(String.format("Deleting user profile: %s", username));

        userProfileService.deleteByUsername(username);
    }
}
