package com.xebialabs.xlrelease.environments.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments.{Application, ApplicationSource}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.ApplicationPersistence.{copyApplication, prepareIdsForSerialization}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.EnvironmentPersistence.copyEnvironments
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.ApplicationRow.{applicationListResultSetExtractor, applicationResultSetExtractor}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.ApplicationSearchResultRow.applicationSearchResultSetExtractor
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.{ApplicationRow, ApplicationSearchResultRow}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema.{APPLICATIONS, APPLICATION_TO_ENVIRONMENTS}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.ENVIRONMENTS
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, PersistenceSupport, SecurablePersistence}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository
import org.springframework.util.CollectionUtils

import java.util.Collections
import scala.jdk.CollectionConverters._
import scala.util.Try

object ApplicationPersistence {
  def prepareIdsForSerialization(application: Application)(implicit ciIdService: CiIdService): Unit = {
    prepareId(application, () => createPersistedId[Application])
    Option(application.getApplicationSource).foreach(s =>
      prepareId(s, () => createPersistedId[ApplicationSource])
    )
    Option(application.getEnvironments).getOrElse(Collections.emptyList()).asScala.foreach(EnvironmentPersistence.prepareIdsForSerialization)
  }

  def copyApplication(app: Application): Application = {
    val applicationCopy = cloneCi(app)
    Option(app.getApplicationSource).foreach(s => applicationCopy.setApplicationSource(cloneCi(s)))
    applicationCopy.setEnvironments(copyEnvironments(app.getEnvironments.asScala.toList).asJava)
    applicationCopy
  }
}

@Repository
@IsTransactional
class ApplicationPersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                          @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                          implicit val ciIdService: CiIdService,
                                          val environmentPersistence: EnvironmentPersistence,
                                          val folderPersistence: FolderPersistence,
                                          val securablePersistence: SecurablePersistence)
  extends PersistenceSupport {

  private val STMT_INSERT_APPLICATIONS =
    s"""|INSERT INTO ${APPLICATIONS.TABLE}
        |   ( ${APPLICATIONS.CI_UID}
        |   , ${APPLICATIONS.FOLDER_UID}
        |   , ${APPLICATIONS.CORRELATION_UID}
        |   , ${APPLICATIONS.ID}
        |   , ${APPLICATIONS.TITLE}
        |   , ${APPLICATIONS.CONTENT}
        |   , ${APPLICATIONS.TENANT_ID}
        |   )
        | VALUES
        |   ( :${APPLICATIONS.CI_UID}
        |   , :${APPLICATIONS.FOLDER_UID}
        |   , :${APPLICATIONS.CORRELATION_UID}
        |   , :${APPLICATIONS.ID}
        |   , :${APPLICATIONS.TITLE}
        |   , :${APPLICATIONS.CONTENT}
        |   , :${APPLICATIONS.TENANT_ID}
        |   )
        """.stripMargin

  private val STMT_EXISTS_BY_TITLE_IGNORECASE_FOLDER =
    s"""|SELECT COUNT(*) FROM ${APPLICATIONS.TABLE} WHERE
        |  LOWER(${APPLICATIONS.TITLE}) = LOWER(:${APPLICATIONS.TITLE})
        |  AND ${APPLICATIONS.FOLDER_UID} = :${APPLICATIONS.FOLDER_UID}""".stripMargin


  private val STMT_EXISTS_BY_TITLE_IGNORECASE_GLOBAL =
    s"""|SELECT COUNT(*) FROM ${APPLICATIONS.TABLE} WHERE
        |  LOWER(${APPLICATIONS.TITLE}) = LOWER(:${APPLICATIONS.TITLE})
        |  AND ${APPLICATIONS.FOLDER_UID} IS NULL""".stripMargin

  def insert(application: Application): CiId = {
    sanitizeApplicationInput(application)

    val exists = Option(application.getFolderId).map(id =>
      sqlQuery(
        STMT_EXISTS_BY_TITLE_IGNORECASE_FOLDER,
        params(APPLICATIONS.TITLE -> application.getTitle, APPLICATIONS.FOLDER_UID -> folderPersistence.getUid(id)),
        _.getInt(1) > 0
      ).head
    ).getOrElse(
      sqlQuery(
        STMT_EXISTS_BY_TITLE_IGNORECASE_GLOBAL,
        params(APPLICATIONS.TITLE -> application.getTitle),
        _.getInt(1) > 0
      ).head
    )
    if (exists) {
      throw new IllegalArgumentException(alreadyExistsMessage("Application", application.getTitle, Option(application.getFolderId)))
    }

    val environmentUids = getEnvironmentUids(application)

    val ciUid = securablePersistence.insert()
    val applicationCopy = copyApplication(application)
    prepareIdsForSerialization(applicationCopy)
    val content = serialize(applicationCopy)
    val displayedAppId = toDisplayId(applicationCopy.getId)
    try {
      sqlExecWithContent(
        STMT_INSERT_APPLICATIONS,
        params(
          APPLICATIONS.CI_UID -> ciUid,
          APPLICATIONS.FOLDER_UID -> Option(applicationCopy.getFolderId).map(folderPersistence.getUid).orNull,
          APPLICATIONS.ID -> applicationCopy.getId,
          APPLICATIONS.CORRELATION_UID -> applicationCopy.getCorrelationUid,
          APPLICATIONS.TITLE -> applicationCopy.getTitle,
          APPLICATIONS.TENANT_ID -> applicationCopy.getTenantId),
        APPLICATIONS.CONTENT -> content,
        identity
      )
    } catch {
      case ex: DuplicateKeyException => throw new IllegalArgumentException(s"Application with ID '$displayedAppId' already exists", ex)
    }

    insertAppToEnvReferences(ciUid, environmentUids.valuesIterator.toSet)

    displayedAppId
  }

  private val STMT_UPDATE_APPLICATIONS =
    s"""|UPDATE ${APPLICATIONS.TABLE}
        | SET
        |  ${APPLICATIONS.TITLE} = :${APPLICATIONS.TITLE},
        |  ${APPLICATIONS.CONTENT} = :${APPLICATIONS.CONTENT}
        | WHERE
        |  ${APPLICATIONS.CI_UID} = :${APPLICATIONS.CI_UID}
       """.stripMargin

  private val STMT_EXISTS_ANOTHER_APP_WITH_TITLE_GLOBAL =
    s"""|$STMT_EXISTS_BY_TITLE_IGNORECASE_GLOBAL
        |AND ${APPLICATIONS.CI_UID} <> :${APPLICATIONS.CI_UID}
       """.stripMargin

  private val STMT_EXISTS_ANOTHER_APP_WITH_TITLE_FOLDER =
    s"""|$STMT_EXISTS_BY_TITLE_IGNORECASE_FOLDER
        |AND ${APPLICATIONS.CI_UID} <> :${APPLICATIONS.CI_UID}
       """.stripMargin

  def update(application: Application): Unit = {
    val applicationUid = findUidById(application.getId)
      .getOrElse(throw new NotFoundException(s"Application [${application.getId}] not found"))

    sanitizeApplicationInput(application)


    val existsAnotherWithSameTitle = Option(application.getFolderId).map(id =>
      sqlQuery(
        STMT_EXISTS_ANOTHER_APP_WITH_TITLE_FOLDER,
        params(APPLICATIONS.TITLE -> application.getTitle, APPLICATIONS.FOLDER_UID -> folderPersistence.getUid(id), APPLICATIONS.CI_UID -> applicationUid),
        _.getInt(1) > 0
      ).head
    ).getOrElse(
      sqlQuery(
        STMT_EXISTS_ANOTHER_APP_WITH_TITLE_GLOBAL,
        params(APPLICATIONS.TITLE -> application.getTitle, APPLICATIONS.CI_UID -> applicationUid),
        _.getInt(1) > 0
      ).head
    )
    if (existsAnotherWithSameTitle) {
      throw new IllegalArgumentException(alreadyExistsMessage("Application", application.getTitle, Option(application.getFolderId)))
    }

    updateAppToEnvReferences(application, applicationUid)
    val applicationCopy = copyApplication(application)
    prepareIdsForSerialization(applicationCopy)

    sqlExecWithContent(
      STMT_UPDATE_APPLICATIONS,
      params(
        APPLICATIONS.CI_UID -> applicationUid,
        APPLICATIONS.TITLE -> applicationCopy.getTitle
      ),
      APPLICATIONS.CONTENT -> serialize(applicationCopy),
      checkCiUpdated(applicationCopy.getId)
    )
  }

  private val STMT_DELETE_APPLICATIONS_BY_ID =
    s"""|DELETE FROM ${APPLICATIONS.TABLE}
        | WHERE ${APPLICATIONS.ID} = :${APPLICATIONS.ID}
       """.stripMargin

  private val STMT_EXISTS_WITH_ENV =
    s"""|SELECT a.${APPLICATIONS.TITLE} FROM ${APPLICATION_TO_ENVIRONMENTS.TABLE} ae
        | JOIN ${ENVIRONMENTS.TABLE} e ON ae.${APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID} = e.${ENVIRONMENTS.CI_UID}
        | JOIN ${APPLICATIONS.TABLE} a ON ae.${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID} = a.${APPLICATIONS.CI_UID}
        | WHERE e.${ENVIRONMENTS.ID} = :${ENVIRONMENTS.ID}
       """.stripMargin

  def delete(applicationId: CiId): Try[Boolean] =
    sqlExec(STMT_DELETE_APPLICATIONS_BY_ID, params(APPLICATIONS.ID -> toPersistedId(applicationId)), ps => Try(ps.execute()))

  private val BASE_STMT_SELECT_APPLICATION_ROW =
    s"""|SELECT app.${APPLICATIONS.CI_UID} ${APPLICATIONS.CI_UID},
        |app.${APPLICATIONS.ID} ${APPLICATIONS.ID},
        |app.${APPLICATIONS.CONTENT} ${APPLICATIONS.CONTENT},
        |folder.${FOLDERS.FOLDER_ID} ${FOLDERS.FOLDER_ID},
        |folder.${FOLDERS.FOLDER_PATH} ${FOLDERS.FOLDER_PATH}
        |FROM ${APPLICATIONS.TABLE} app
        |LEFT JOIN ${FOLDERS.TABLE} folder
        | ON app.${APPLICATIONS.FOLDER_UID} = folder.${FOLDERS.CI_UID}""".stripMargin

  def findById(applicationId: CiId): Option[ApplicationRow] = {
    val stmt =
      s"""|$BASE_STMT_SELECT_APPLICATION_ROW
          |WHERE app.${APPLICATIONS.ID} = :${APPLICATIONS.ID}""".stripMargin
    sqlQuery(stmt, params(APPLICATIONS.ID -> toPersistedId(applicationId)), applicationResultSetExtractor())
  }

  def findByTitle(title: String): Option[ApplicationRow] = {
    val stmt =
      s"""|$BASE_STMT_SELECT_APPLICATION_ROW
          |WHERE LOWER(app.${APPLICATIONS.TITLE}) = LOWER(:${APPLICATIONS.TITLE})""".stripMargin
    sqlQuery(stmt, params(APPLICATIONS.TITLE -> title), applicationResultSetExtractor())
  }

  def findByTitleAndFolder(title: String, folderMaybe: Option[String]): Option[ApplicationRow] = {
    folderMaybe.map(folderId => {
      val stmt =
        s"""|$BASE_STMT_SELECT_APPLICATION_ROW
            |WHERE LOWER(app.${APPLICATIONS.TITLE}) = LOWER(:${APPLICATIONS.TITLE})
            | AND app.${APPLICATIONS.FOLDER_UID} = :${APPLICATIONS.FOLDER_UID}""".stripMargin
      sqlQuery(stmt,
        params(
          APPLICATIONS.TITLE -> title,
          APPLICATIONS.FOLDER_UID -> folderPersistence.getUid(folderId)
        ),
        applicationResultSetExtractor())
    }).getOrElse {
      val stmt =
        s"""|$BASE_STMT_SELECT_APPLICATION_ROW
            |WHERE LOWER(app.${APPLICATIONS.TITLE}) = LOWER(:${APPLICATIONS.TITLE})
            | AND app.${APPLICATIONS.FOLDER_UID} IS NULL""".stripMargin
      sqlQuery(stmt,
        params(APPLICATIONS.TITLE -> title),
        applicationResultSetExtractor())
    }
  }

  def findAppsWithEnv(envId: CiId): List[String] = {
    val apps = sqlQuery(
      STMT_EXISTS_WITH_ENV,
      params(ENVIRONMENTS.ID -> toPersistedId(envId)),
      _.getString(1)
    )
    apps.toList
  }

  def findInFolderByCorrelationId(folderId: String, correlationId: String): Option[ApplicationRow] = {
    val stmt =
      s"""|$BASE_STMT_SELECT_APPLICATION_ROW
          |WHERE app.${APPLICATIONS.CORRELATION_UID} = :${APPLICATIONS.CORRELATION_UID}
          | AND app.${APPLICATIONS.FOLDER_UID} = :${APPLICATIONS.FOLDER_UID}""".stripMargin

    sqlQuery(stmt,
      params(
        APPLICATIONS.CORRELATION_UID -> correlationId,
        APPLICATIONS.FOLDER_UID -> folderPersistence.getUid(folderId)
      ),
      applicationResultSetExtractor())
  }

  private val STMT_TENANT_APPLICATION_COUNT =
    s"""|SELECT COUNT(1)
        | FROM ${APPLICATIONS.TABLE}
        | WHERE ${APPLICATIONS.TENANT_ID} = :${APPLICATIONS.TENANT_ID}""".stripMargin

  def tenantApplicationCount(tenantId: String): Int = {
    namedTemplate.queryForObject(STMT_TENANT_APPLICATION_COUNT, paramSource(APPLICATIONS.TENANT_ID -> tenantId), classOf[Int])
  }

  def findUidById(applicationId: CiId): Option[CiUid] = {
    val stmt =
      s"""|SELECT ${APPLICATIONS.CI_UID}
          |FROM ${APPLICATIONS.TABLE}
          |WHERE ${APPLICATIONS.ID} = :${APPLICATIONS.ID}""".stripMargin
    sqlQuery(stmt, params(APPLICATIONS.ID -> toPersistedId(applicationId)), rs => CiUid(rs.getInt(APPLICATIONS.CI_UID))).headOption
  }

  def getUidsByIds(applicationIds: Iterable[CiId]): Map[CiId, CiUid] = {
    val stmt =
      s"""|SELECT ${APPLICATIONS.CI_UID}, ${APPLICATIONS.ID}
          |FROM ${APPLICATIONS.TABLE}
          |WHERE ${APPLICATIONS.ID} IN (:applicationIds)
       """.stripMargin
    sqlQuery(stmt,
      params("applicationIds" -> applicationIds.map(toPersistedId).asJava),
      rs => rs.getCiId(APPLICATIONS.ID) -> CiUid(rs.getInt(APPLICATIONS.CI_UID))
    ).toMap
  }

  def search(sqlWithParameters: SqlWithParameters): Seq[ApplicationSearchResultRow] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query(sql, applicationSearchResultSetExtractor(), params: _*)
  }

  private def getEnvironmentUids(application: Application): Map[CiId, CiUid] = {
    if (CollectionUtils.isEmpty(application.getEnvironments)) {
      Map.empty[CiId, CiUid]
    } else {
      val environmentIds = application.getEnvironments.asScala.map(_.getId).toSet
      val environmentIdsToUids = environmentPersistence.getUidsByIds(environmentIds)
      val environmentDiff = environmentIds.map(toDisplayId).diff(environmentIdsToUids.keySet)
      if (environmentDiff.nonEmpty) {
        throw new NotFoundException(s"Environments [${environmentDiff.mkString(", ")}] not found")
      }
      environmentIdsToUids
    }
  }

  def fetchApplications(applicationIds: List[CiId]): Seq[ApplicationRow] = {
    val stmt =
      s"""|$BASE_STMT_SELECT_APPLICATION_ROW
          |WHERE app.${APPLICATIONS.ID} IN (:applicationIds)""".stripMargin
    sqlQuery(stmt,
      params(
        "applicationIds" -> applicationIds.map(toPersistedId).asJava
      ),
      applicationListResultSetExtractor())
  }

  def fetchApplicationsByFolderId(folderId: String): Seq[ApplicationRow] = {
    val stmt =
      s"""|$BASE_STMT_SELECT_APPLICATION_ROW
          |WHERE app.${APPLICATIONS.FOLDER_UID} = :${APPLICATIONS.FOLDER_UID}""".stripMargin
    sqlQuery(stmt,
      params(
        APPLICATIONS.FOLDER_UID -> folderPersistence.getUid(folderId)
      ),
      applicationListResultSetExtractor())
  }


  private def insertAppToEnvReferences(applicationUid: CiUid, envUids: Set[CiUid]): Unit = {
    val insertAppTpEnvStmt =
      s"""INSERT INTO ${APPLICATION_TO_ENVIRONMENTS.TABLE} (
         |${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID},
         |${APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID}
         |)
         |VALUES (
         |:${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID},
         |:${APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID}
         |)
       """.stripMargin

    sqlBatch(insertAppTpEnvStmt,
      envUids.map { envUid =>
        params(
          APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID -> applicationUid,
          APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID -> envUid
        )
      }
    )
  }

  private def updateAppToEnvReferences(application: Application, applicationUid: CiUid): Unit = {
    val envIdsToUids = getEnvironmentUids(application)

    val deleteAppToEnvRefsStmt =
      s"""
         |DELETE FROM ${APPLICATION_TO_ENVIRONMENTS.TABLE}
         | WHERE
         | ${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID} = :${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID}
       """.stripMargin
    sqlExec(deleteAppToEnvRefsStmt, params(APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID -> applicationUid), _.execute())
    insertAppToEnvReferences(applicationUid, envIdsToUids.valuesIterator.toSet)
  }

  private def sanitizeApplicationInput(application: Application): Unit = {
    application.setTitle(application.getTitle.trimAndTruncate(ApplicationSchema.TITLE_LENGTH))
  }
}
