package com.xebialabs.xlrelease.environments.initialize

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.environments.{Environment, EnvironmentLabel, EnvironmentStage}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.ColumnAliases
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentLabelSchema.ENV_LABELS
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.{ENVIRONMENTS, ENV_TO_LABEL}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentStageSchema.ENV_STAGES
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, ParallelSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor}
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import scala.collection.mutable
import scala.jdk.CollectionConverters._


@Component
class XlRelease243EnvironmentAddJsonContentUpgrader(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                    @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                    @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate)
  extends Upgrade with BatchSupport with ParallelSupport with PersistenceSupport with TransactionSupport with Logging {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "24.3.0#0")

  override def doUpgrade(): Boolean = {
    logger.info("Updating Environments with CI JSON content")
    val environments = findEnvironmentRows
    doInBatch(environments) { batch =>
      doInTransaction {
        doInParallel(batch.items) {
          case (environmentUid, environment) =>
            updateEnvironmentContent(environmentUid, environment)
        }
      }
    }
    logger.info("Finished updating Environments with CI JSON content")
    true
  }

  private val STMT_UPDATE_ENVIRONMENTS =
    s"""|UPDATE ${ENVIRONMENTS.TABLE}
        | SET
        |  ${ENVIRONMENTS.CONTENT} = :${ENVIRONMENTS.CONTENT}
        | WHERE
        |  ${ENVIRONMENTS.CI_UID} = :${ENVIRONMENTS.CI_UID}
       """.stripMargin

  private def updateEnvironmentContent(environmentUid: CiUid, environment: Environment): Unit = {
    sqlExecWithContent(
      STMT_UPDATE_ENVIRONMENTS,
      params(
        ENVIRONMENTS.CI_UID -> environmentUid,
      ),
      ENVIRONMENTS.CONTENT -> serialize(environment),
      checkCiUpdated(environment.getId)
    )
  }

  private val STMT_FIND_ENVS =
    s"""
       |SELECT
       |  ${ENVIRONMENTS.CI_UID},
       |  ${ENVIRONMENTS.ID},
       |  ${ENVIRONMENTS.ENV_STAGE_UID},
       |  ${ENVIRONMENTS.DESCRIPTION},
       |  ${ENVIRONMENTS.TITLE}
       |FROM
       |  ${ENVIRONMENTS.TABLE}
     """.stripMargin

  private def findEnvironmentRows: Vector[(CiUid, Environment)] = {
    sqlQuery(STMT_FIND_ENVS, params(), environmentMapper())
  }

  private def environmentMapper(): ResultSetExtractor[Vector[(CiUid, Environment)]] = rs => {
    val envs = mutable.Set.empty[(CiUid, Environment)]

    while(rs.next()) {
      val env = new Environment
      env.setTitle(rs.getString(ENVIRONMENTS.TITLE))
      env.setId(rs.getString(ENVIRONMENTS.ID))
      env.setDescription(rs.getString(ENVIRONMENTS.DESCRIPTION))
      val ciuid = rs.getInt(ENVIRONMENTS.CI_UID)
      val stageUid = rs.getInt(ENVIRONMENTS.ENV_STAGE_UID)
      val stage = findStageForEnvironment(stageUid)
      env.setStage(stage)
      val labels = findLabelsForEnvironment(ciuid).toList.asJava
      env.setLabels(labels)
      envs.add((ciuid, env))
    }
    envs.toVector
  }

  private val STMT_FIND_STAGE =
    s"""
       |SELECT
       |  ${ENV_STAGES.CI_UID},
       |  ${ENV_STAGES.ID},
       |  ${ENV_STAGES.TITLE}
       |FROM
       |  ${ENV_STAGES.TABLE}
       |WHERE ${ENV_STAGES.CI_UID} = :stageUid
     """.stripMargin

  private def findStageForEnvironment(stageUid: CiUid): EnvironmentStage =
    sqlQuery(STMT_FIND_STAGE, params("stageUid" -> stageUid), stageMapper()).getOrElse {
      throw new IllegalStateException(s"Stage row [$stageUid] not found")
    }

  private def stageMapper(): ResultSetExtractor[Option[EnvironmentStage]] = rs =>
    if (rs.next()) {
      val stage = new EnvironmentStage
      stage.setTitle(rs.getString(ENV_STAGES.TITLE))
      stage.setId(rs.getString(ENV_STAGES.ID))
      Some(stage)
    } else {
      None
    }

  private val STMT_FIND_LABELS_FOR_ENV =
    s"""
       |SELECT
       |  label.${ENV_LABELS.CI_UID} ${ColumnAliases.EnvLabels.CI_UID},
       |  label.${ENV_LABELS.ID} ${ColumnAliases.EnvLabels.ID},
       |  label.${ENV_LABELS.TITLE} ${ColumnAliases.EnvLabels.TITLE},
       |  label.${ENV_LABELS.COLOR} ${ColumnAliases.EnvLabels.COLOR}
       |FROM
       |  ${ENVIRONMENTS.TABLE} env
       |LEFT JOIN ${ENV_TO_LABEL.TABLE} envlabel
       |  on env.${ENVIRONMENTS.CI_UID} = envlabel.${ENV_TO_LABEL.ENVIRONMENT_UID}
       |LEFT JOIN ${ENV_LABELS.TABLE} label
       |  on envlabel.${ENV_TO_LABEL.LABEL_UID} = label.${ENV_LABELS.CI_UID}
       |WHERE env.${ENVIRONMENTS.CI_UID} = :envId
     """.stripMargin

  private def findLabelsForEnvironment(envId: CiUid): Vector[EnvironmentLabel] = {
    sqlQuery(STMT_FIND_LABELS_FOR_ENV, params("envId" -> envId), labelMapper())
  }

  private def labelMapper(): ResultSetExtractor[Vector[EnvironmentLabel]] = rs => {
    val labels = mutable.Set.empty[EnvironmentLabel]
    while(rs.next()) {
      val label = new EnvironmentLabel
      label.setTitle(rs.getString(ColumnAliases.EnvLabels.TITLE))
      label.setId(rs.getString(ColumnAliases.EnvLabels.ID))
      label.setColor(rs.getString(ColumnAliases.EnvLabels.COLOR))
      labels.add(label)
    }
    labels.toVector
  }
}
