import React, { useState } from 'react';
import includes from 'lodash/includes';
import sortBy from 'lodash/sortBy';
import defaults from 'lodash/defaults';
import EnvironmentSidebarItem from '../../../shared/environment/environment-sidebar-item';

import './sidebar-environment-list.less';
import { Environment, Stage } from '../../../../application-list.types';
import { DotInputSelect, DotInputText, DotThemeProvider } from '@digital-ai/dot-components';

export interface SidebarEnvironmentListProps {
    environments: Array<Environment>;
    onClickEnvironment: (environment: Environment) => void;
    stages: Array<Stage>;
}
const filterOptions = { placeholder: 'Filter environment name...' };

export const SidebarEnvironmentList = ({ environments = [], onClickEnvironment, stages = [] }: SidebarEnvironmentListProps) => {
    const [stageSelected, setStageSelected] = useState('');
    const [environmentFilter, setEnvironmentFilter] = useState('');

    const onStageChange = (stage: string) => {
        setStageSelected(stage);
    };

    const onNameChange = (name: string) => {
        setEnvironmentFilter(name);
    };

    const onEnvironmentClick = (environment: Environment) => {
        onClickEnvironment(environment);
    };

    const defaultOption = { option: 'Filter by stage...' };
    const filterEnvironments = (environment: Environment) => {
        const stage = stageSelected === defaultOption.option ? '' : stageSelected.toLowerCase();
        defaults(stageSelected, { option: '' });
        return includes(environment.title.toLowerCase(), environmentFilter.toLowerCase()) && includes(environment.stage.title.toLowerCase(), stage);
    };

    const renderEnvironments = () => {
        return environments
            .filter((env) => filterEnvironments(env))
            .map((env) => <EnvironmentSidebarItem environment={env} isClickable={true} key={env.id} onClick={onEnvironmentClick} />);
    };

    const stageOptions = sortBy(
        stages.map((stage) => {
            return { option: stage.title };
        }),
        ['option'],
    );

    return (
        <div className="application-form-sidebar">
            <div className="application-form-environment-sidebar">
                <DotThemeProvider>
                    <div className="header">
                        <h4 className="page-title">Environments</h4>
                        <DotInputSelect
                            defaultValue={defaultOption.option}
                            id={'stageSelector'}
                            name={'stageSelector'}
                            onChange={(e) => onStageChange(e.target.value)}
                            options={[defaultOption, ...stageOptions]}
                        />
                        <DotInputText
                            id={'environmentFilter'}
                            name={'environmentFilter'}
                            onChange={(e) => {
                                onNameChange(e.target.value);
                            }}
                            placeholder={filterOptions.placeholder}
                        />
                    </div>
                    <div className="environment-sidebar-list">{renderEnvironments()}</div>
                </DotThemeProvider>
            </div>
        </div>
    );
};
