package com.xebialabs.xlrelease.environments.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.filter.ReservationFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.environments._
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.{ColumnAliases, EnvironmentsWithReservationsSqlBuilder}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.EnvironmentReservationRow
import com.xebialabs.xlrelease.environments.repository.sql.persistence.{ApplicationPersistence, EnvironmentPersistence, EnvironmentReservationPersistence, toDisplayId}
import com.xebialabs.xlrelease.environments.repository.{EnvironmentLabelRepository, EnvironmentReservationRepository, EnvironmentStageRepository}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Repository

import java.util.Date
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@IsTransactional
@Repository
class SqlEnvironmentReservationRepository @Autowired()(@Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                       implicit val ciIdService: CiIdService,
                                                       implicit val repositoryAdapter: SqlRepositoryAdapter,
                                                       implicit val applicationPersistence: ApplicationPersistence,
                                                       implicit val environmentPersistence: EnvironmentPersistence,
                                                       implicit val environmentStageRepository: EnvironmentStageRepository,
                                                       implicit val environmentLabelRepository: EnvironmentLabelRepository,
                                                       environmentReservationPersistence: EnvironmentReservationPersistence)
  extends EnvironmentReservationRepository {

  @IsReadOnly
  override def search(filters: ReservationFilters, page: Page): Map[Environment, Seq[EnvironmentReservation]] = {
    val query = EnvironmentsWithReservationsSqlBuilder()
      .orderBy(ColumnAliases.Environments.CI_UID)
      .limitAndOffset(page.resultsPerPage, page.offset)

    Option(filters).foreach { f =>
      Option(f.getEnvironmentTitle).foreach(query.withEnvironmentTitleLike)
      Option(f.getStages).foreach(stages => query.withStageTitles(stages.asScala))
      Option(f.getApplications).foreach(apps => query.withApplicationTitles(apps.asScala))
      Option(f.getLabels).foreach(labels => query.withLabelTitles(labels.asScala))
      Option(f.getFrom).foreach(query.from)
      Option(f.getTo).foreach(query.to)
    }

    environmentReservationPersistence.search(query.build()).map {
      case (envId, reservationRows) =>
        mapEnvironmentReservations(envId, reservationRows)
    }
  }

  @IsReadOnly
  override def findById(environmentReservationId: String): EnvironmentReservation = {
    implicit val cachedEnvs: mutable.Map[CiId, Environment] = mutable.Map.empty[CiId, Environment]
    implicit val cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty[CiId, EnvironmentStage]
    implicit val cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty[CiId, EnvironmentLabel]
    environmentReservationPersistence.findById(environmentReservationId).map(
        res => mapEnvironmentReservationFromRow(mapEnvironment(res.environmentId), res))
      .getOrElse(throw new NotFoundException(s"Environment reservation [$environmentReservationId] not found"))
  }

  @IsReadOnly
  override def getAllForEnvironmentId(environmentId: String): Seq[EnvironmentReservation] = {
    val tmpEnvironment = new Environment().getType.getDescriptor.newInstance[Environment](environmentId)
    val query = EnvironmentsWithReservationsSqlBuilder().withEnvironmentId(environmentId)
    environmentReservationPersistence.search(query.build()).map {
        case (envRow, reservationRows) =>
          mapEnvironmentReservations(envRow, reservationRows)
      }
      .getOrElse(tmpEnvironment, Seq.empty[EnvironmentReservation])
  }

  override def create(environmentReservation: EnvironmentReservation): EnvironmentReservation = {
    val resId = environmentReservationPersistence.insert(environmentReservation)
    findById(resId)
  }

  override def update(environmentReservation: EnvironmentReservation): EnvironmentReservation = {
    environmentReservationPersistence.update(environmentReservation)
    findById(environmentReservation.getId)
  }

  override def delete(environmentReservationId: String): Unit = {
    environmentReservationPersistence.delete(environmentReservationId)
  }

  override def existsByEnvironmentIdAndApplicationIdAndDate(environmentId: String, applicationId: String, date: Date): Boolean = {
    environmentReservationPersistence.exists(environmentId, applicationId, date)
  }

  override def findNearestComingByEnvironmentIdAndApplicationIdAndDate(environmentId: String, applicationId: String, date: Date): Option[Date] = {
    environmentReservationPersistence.findNearestComing(environmentId, applicationId, date)
  }

  private def mapEnvironmentReservations(envId: CiId,
                                         reservationRows: Seq[EnvironmentReservationRow])
                                        (implicit cachedApps: mutable.Map[CiId, Application] = mutable.Map.empty[CiId, Application],
                                         cachedEnvs: mutable.Map[CiId, Environment] = mutable.Map.empty[CiId, Environment],
                                         cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty[CiId, EnvironmentStage],
                                         cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty[CiId, EnvironmentLabel])
  : (Environment, Seq[EnvironmentReservation]) = {
    val env = mapEnvironment(envId)
    env -> reservationRows.map(row => {
      mapEnvironmentReservationFromRow(env, row)
    })
  }

  private def mapEnvironmentReservationFromRow(env: Environment, row: EnvironmentReservationRow)(
    implicit cachedApps: mutable.Map[CiId, Application] = mutable.Map.empty[CiId, Application],
    cachedEnvs: mutable.Map[CiId, Environment] = mutable.Map.empty[CiId, Environment],
    cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty[CiId, EnvironmentStage],
    cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty[CiId, EnvironmentLabel]): EnvironmentReservation = {

    val apps = row.applicationIds.map(mapApplication).toList.asJava
    val envRes = new EnvironmentReservation()
    envRes.setId(toDisplayId(row.id))
    envRes.setEnvironment(env)
    envRes.setApplications(apps)
    envRes.setStartDate(row.from)
    envRes.setEndDate(row.to)
    envRes.setNote(row.note)
    envRes
  }
}
