import React, { useEffect, useState } from 'react';
import isEmpty from 'lodash/isEmpty';
import isNil from 'lodash/isNil';
import { TwitterPicker } from 'react-color';

import { itemAlreadyDefined } from '@xlr-ui/app/react/utils/validators';
import colors from '@xlr-ui/app/js/colors';

import './../../../styles/modal.less';
import './label-modal-form-component.less';
import { DotButton, DotCheckbox, DotDialog, DotIcon, DotInputText, DotTypography } from '@digital-ai/dot-components';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { getLabelsList } from '../ducks/environment-list.selectors';
import { labelsState } from '../ducks/labels.reducer';
import { initEnvLabel } from './label-list-component';
import { Label } from '../../../application-list.types';

const titleFieldMetadata = {
    label: 'Name',
    placeholder: 'Add...',
    description: 'Name of the label',
    required: true,
};

const errorMsg = 'Label already exists';
const defaultColorList = [colors.blue, colors.green, colors.orange, colors.red, colors.gray, colors.purple];

const { searchLabels } = labelsState.actions;

export interface LabelModalFormProps {
    labelModel: Label;
    modalIsOpen: boolean;
    onCancelModal: () => void;
    onConfirmModal: (label: Label, another: boolean) => void;
    onOpenDelete: () => void;
}

export const LabelModalFormComponent = ({ labelModel, modalIsOpen, onCancelModal, onConfirmModal, onOpenDelete }: LabelModalFormProps) => {
    const envLabelList = useAppSelector(getLabelsList);
    const dispatch = useAppDispatch();

    const [isValid, setIsValid] = useState(true);
    const [isPristine, setPristine] = useState(true);
    const [localLabelModel, setLocalLabelModel] = useState(initEnvLabel);
    const [createAnother, setCreateAnother] = React.useState(false);
    const [displayColorPicker, setDisplayColorPicker] = React.useState(false);
    const [colorIsChanged, setColorIsChanged] = useState(false);

    useEffect(() => {
        dispatch(searchLabels(''));
        setLocalLabelModel(labelModel);
        if (isEmpty(localLabelModel.color)) {
            setLocalLabelModel({
                ...localLabelModel,
                color: defaultColorList[0],
            });
        }
    }, []);

    useEffect(() => {
        setLocalLabelModel(labelModel);
        if (isEmpty(labelModel.color)) {
            setLocalLabelModel({
                ...labelModel,
                color: defaultColorList[0],
            });
        }
    }, [labelModel]);

    const closeModal = () => {
        cancelForm();
        resetForm();
    };
    const cancelForm = () => {
        setCreateAnother(false);
        onCancelModal();
    };
    const resetForm = () => {
        setColorIsChanged(false);
        setLocalLabelModel(initEnvLabel);
    };

    const submitModal = () => {
        onConfirmModal(localLabelModel, createAnother);
        setCreateAnother(false);
        setLocalLabelModel({
            ...initEnvLabel,
            color: localLabelModel.color,
        });
    };

    const onModelChange = (title: string) => {
        if (itemAlreadyDefined(envLabelList, labelModel.id, 'title', title) || isEmpty(title)) {
            setIsValid(false);
        } else if (title !== labelModel.title) {
            setLocalLabelModel({
                ...localLabelModel,
                title,
            });
            setIsValid(true);
            setPristine(false);
        }
        if (title === labelModel.title) {
            setPristine(true);
        }
    };

    const onColorChangeComplete = (color: { hex: string }) => {
        if (localLabelModel.color !== color.hex) {
            setColorIsChanged(true);
        }
        setDisplayColorPicker(false);
        setLocalLabelModel({
            ...localLabelModel,
            color: color.hex,
        });
    };

    const onDelete = () => {
        onOpenDelete();
    };

    const renderColorPickerPopover = () => {
        return displayColorPicker ? (
            <div className="color-picker-popover">
                <div
                    className="cover"
                    onClick={() => {
                        setDisplayColorPicker(false);
                    }}
                />
                <TwitterPicker color={localLabelModel.color} colors={defaultColorList} onChangeComplete={onColorChangeComplete} width="204" />
            </div>
        ) : null;
    };
    const renderAdditionalButtons = () => {
        return labelModel.id ? (
            <DotButton className={'additional-delete'} fullWidth={false} onClick={onDelete} startIcon={<DotIcon iconId="delete" />} type="destructive">
                Delete label
            </DotButton>
        ) : (
            <DotCheckbox checked={createAnother} label={'Create another'} onChange={(e) => setCreateAnother(e.target.checked)} />
        );
    };

    const modalTitle = isNil(labelModel.id) ? 'New label' : 'Edit label';

    return (
        <DotDialog
            className="xl-default-modal"
            closeOnSubmit={false}
            onCancel={closeModal}
            onSubmit={submitModal}
            open={modalIsOpen}
            submitButtonProps={{ label: labelModel.id ? 'Save' : 'Create', disabled: !isValid || (isPristine && !colorIsChanged), 'data-testid': 'save-btn' }}
            title={<DotTypography variant="h1">{modalTitle}</DotTypography>}
        >
            <div className="environment-label-form">
                <DotInputText
                    className={'label-form-input'}
                    error={!isValid}
                    helperText={isValid ? '' : errorMsg}
                    id={'modal-edit-label-input'}
                    label={titleFieldMetadata.label}
                    maxLength={255}
                    name={'modal-edit-label-input'}
                    onChange={(e) => onModelChange(e.target.value)}
                    persistentLabel={true}
                    placeholder={titleFieldMetadata.placeholder}
                    value={localLabelModel.title}
                />
                <div className="xl-react-components">
                    <div className="xl-react-components-label">
                        <label>
                            Color<span className="required">*</span>
                        </label>
                    </div>
                    <div className="xl-react-components-input">
                        <div className="swatch" onClick={() => setDisplayColorPicker(!displayColorPicker)}>
                            <div className="color" style={{ background: localLabelModel.color }} />
                        </div>
                        {renderColorPickerPopover()}
                    </div>
                </div>
                <div className={'additional-dialog-buttons'}>{renderAdditionalButtons()}</div>
            </div>
        </DotDialog>
    );
};
