package com.xebialabs.xlrelease.environments.service
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.xlrelease.api.v1.filter.ApplicationFilters
import com.xebialabs.xlrelease.domain.environments.Application
import com.xebialabs.xlrelease.environments.events.{ApplicationCreatedEvent, ApplicationDeletedEvent, ApplicationUpdatedEvent}
import com.xebialabs.xlrelease.environments.repository.ApplicationRepository
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Service
class ApplicationService @Autowired()(applicationRepository: ApplicationRepository,
                                      val eventBus: XLReleaseEventBus) {
  @Timed
  def searchApplications(applicationFilters: ApplicationFilters): JList[Application] = {
    applicationRepository.searchApplications(applicationFilters, defaultPage).asJava
  }

  @Timed
  def findApplicationById(applicationId: String): Application = {
    val application = applicationRepository.findApplicationById(applicationId)
    application
  }

  @Timed
  def createApplication(application: Application): Application = {
    validateTitle(application)
    val createdApplication = applicationRepository.createApplication(application)
    eventBus.publish(ApplicationCreatedEvent(createdApplication))
    createdApplication
  }

  @Timed
  def updateApplication(application: Application): Application = {
    checkArgument(hasText(application.getId), "ID is required")
    validateTitle(application)
    val updatedApplication = applicationRepository.updateApplication(application)
    eventBus.publish(ApplicationUpdatedEvent(updatedApplication))
    updatedApplication
  }

  @Timed
  def deleteApplication(applicationId: String): Unit = {
    val application = applicationRepository.findApplicationById(applicationId)
    applicationRepository.deleteApplication(applicationId)
    eventBus.publish(ApplicationDeletedEvent(application))
  }

  @Timed
  def getApplicationsDeployableOnEnvironment(environmentId: String): JList[Application] = {
    applicationRepository.getApplicationsDeployableOnEnvironment(environmentId).asJava
  }

  @Timed
  def findByEnvironment(environmentId: String): List[String] = {
    applicationRepository.findByEnvironment(environmentId)
  }

  private def validateTitle(application: Application): Unit = {
    checkArgument(hasText(application.getTitle), "Title cannot be blank")
  }
}
