import React, { useEffect, useState } from 'react';
import differenceBy from 'lodash/differenceBy';
import isEmpty from 'lodash/isEmpty';
import { applicationForm } from '../ducks/application-form.reducer';
import { Application, Environment } from '../../../application-list.types';
import { ApplicationEditHeaderComponent } from './application-edit-header-component';
import { ApplicationFormComponent } from './form/application-form-component';
import { SidebarEnvironmentList } from './form/sidebar-environment-list';

import './application-edit-page-component.less';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { getFormApplication, getFormApplications, getFormEnvironments, getFormStages } from '../ducks/application-list.selectors';

const { initForm, onSave, onUpdate, cancelForm, setApplicationFolder } = applicationForm.actions;

export interface ApplicationEditPageProps {
    applicationId: string;
    folderId: string;
}

export const ApplicationEditPageComponent = ({ applicationId, folderId }: ApplicationEditPageProps) => {
    const dispatch = useAppDispatch();

    const [isFormValid, setFormValid] = useState(true);
    const [isFormPristine, setFormPristine] = useState(true);

    useEffect(() => {
        dispatch(initForm({ appId: applicationId, folderId }));
    }, []);

    const application = useAppSelector(getFormApplication);
    const applications = useAppSelector(getFormApplications);
    const environments = useAppSelector(getFormEnvironments);
    const stages = useAppSelector(getFormStages);
    const onSaveAction = () => {
        dispatch(setApplicationFolder(folderId));
        dispatch(onSave(application));
    };

    const onFormValidChange = (valid: boolean) => {
        setFormValid(valid);
    };

    const onApplicationChange = (app: Application) => {
        dispatch(onUpdate({ app, folderId }));
        setFormPristine(false);
    };

    const onAddEnvironment = (environment: Environment) => {
        setFormPristine(false);
        const updatedApp = {
            ...application,
            environments: [...application.environments, environment],
        };
        dispatch(onUpdate({ app: updatedApp, folderId }));
    };

    const onFormCancel = () => {
        dispatch(cancelForm(folderId));
    };

    const renderSidebarContent = () => {
        const filteredEnvironments = differenceBy(environments, application.environments, 'id');

        return <SidebarEnvironmentList environments={filteredEnvironments} onClickEnvironment={onAddEnvironment} stages={stages} />;
    };

    const saveEnabled = !isFormPristine && isFormValid;
    const saveButtonCaption = isEmpty(application) ? 'Create' : 'Save';

    return (
        <div className="application-form-view">
            <ApplicationEditHeaderComponent onCancel={onFormCancel} onSave={onSaveAction} saveCaption={saveButtonCaption} saveEnabled={saveEnabled} />
            <div className={'application-form-content'}>
                {renderSidebarContent()}
                <ApplicationFormComponent
                    applicationList={applications}
                    applicationModel={application}
                    onFormChange={onApplicationChange}
                    onValidationChange={onFormValidChange}
                />
            </div>
        </div>
    );
};
