package com.xebialabs.xlrelease.environments.repository.sql.persistence.builder

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder, SqlWithParameters}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema.APPLICATIONS
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentReservationSchema.{ENV_RES_TO_APP, ENV_RESERVATIONS => ER}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS

import java.util.Date

object ReservationSqlBuilder {
  def apply(aliasPrefix: String = "")(implicit dialect: Dialect): ReservationSqlBuilder = {
    new ReservationSqlBuilder(aliasPrefix)
  }


}

class ReservationSqlBuilder(aliasPrefix: String)(implicit dialect: Dialect) extends SqlBuilder[ReservationSqlBuilder] {
  private val baseResAlias = makeAlias(aliasPrefix, "baseRes")
  private val resToAppAlias = makeAlias(aliasPrefix, "resToA")
  private val appAlias = makeAlias(aliasPrefix, "app")
  private val folderAlias = makeAlias(aliasPrefix, "folder")

  private lazy val baseBuilder = new BaseReservationSqlBuilder(baseResAlias)

  def select(): ReservationSqlBuilder = this

  override def build(): SqlWithParameters = {
    val (innerQuery, innerParams) = baseBuilder.select().build()
    val stmt =
      s"""
         |SELECT
         | $baseResAlias.${ColumnAliases.EnvReservations.ID} ${ColumnAliases.EnvReservations.ID}
         | ,$baseResAlias.${ColumnAliases.EnvReservations.START_DATE} ${ColumnAliases.EnvReservations.START_DATE}
         | ,$baseResAlias.${ColumnAliases.EnvReservations.END_DATE} ${ColumnAliases.EnvReservations.END_DATE}
         | ,$baseResAlias.${ColumnAliases.EnvReservations.NOTE} ${ColumnAliases.EnvReservations.NOTE}
         | ,$baseResAlias.${ColumnAliases.EnvReservations.ENVIRONMENT_UID} ${ColumnAliases.EnvReservations.ENVIRONMENT_UID}
         | ,$appAlias.${APPLICATIONS.ID} ${ColumnAliases.Applications.ID}
         | ,$folderAlias.${FOLDERS.FOLDER_ID} ${ColumnAliases.Applications.FOLDER_ID}
         | ,$folderAlias.${FOLDERS.FOLDER_PATH} ${ColumnAliases.Applications.FOLDER_PATH}
         | ,$appAlias.${APPLICATIONS.TITLE} ${ColumnAliases.Applications.TITLE}
         | ,$appAlias.${APPLICATIONS.CONTENT} ${ColumnAliases.Applications.CONTENT}
         | FROM (${indent(innerQuery)}) $baseResAlias
         | LEFT JOIN ${ENV_RES_TO_APP.TABLE} $resToAppAlias
         |  ON $baseResAlias.${ColumnAliases.EnvReservations.CI_UID} = $resToAppAlias.${ENV_RES_TO_APP.RESERVATION_UID}
         | LEFT JOIN ${APPLICATIONS.TABLE} $appAlias
         |  ON $resToAppAlias.${ENV_RES_TO_APP.APPLICATION_UID} = $appAlias.${APPLICATIONS.CI_UID}
         | LEFT JOIN ${FOLDERS.TABLE} $folderAlias
         |  ON $appAlias.${APPLICATIONS.FOLDER_UID} = $folderAlias.${FOLDERS.CI_UID}
       """.stripMargin
    super.select(stmt).orderBy(ColumnAliases.EnvReservations.START_DATE)
    val fullSql = super.build()
    fullSql._1 -> (innerParams ++ fullSql._2)
  }

  def withApplicationTitles(titles: Iterable[String]): ReservationSqlBuilder = {
    baseBuilder.withApplicationTitles(titles)
    this
  }

  def from(date: Date): ReservationSqlBuilder = {
    baseBuilder.from(date)
    this
  }

  def to(date: Date): ReservationSqlBuilder = {
    baseBuilder.to(date)
    this
  }

  override def limitAndOffset(limit: Long, offset: Long): ReservationSqlBuilder = {
    baseBuilder.limitAndOffset(limit, offset)
    this
  }

  override def newInstance: ReservationSqlBuilder = new ReservationSqlBuilder(aliasPrefix)

  class BaseReservationSqlBuilder(aliasPrefix: String)(implicit dialect: Dialect) extends SqlBuilder[BaseReservationSqlBuilder] {
    private val resAlias = makeAlias(aliasPrefix, "res")
    private val resToAppAlias = makeAlias(aliasPrefix, "resToA")
    private val appAlias = makeAlias(aliasPrefix, "app")

    // todo define dynamic joins for if with environments or without?
    def select(): BaseReservationSqlBuilder = this

    // Many-to-many joins are not returned, they are only used for filtering
    // This is to make filtering work properly
    override def build(): SqlWithParameters = {
      val stmt =
        s"""
           |SELECT DISTINCT
           | $resAlias.${ER.CI_UID} ${ColumnAliases.EnvReservations.CI_UID}
           | ,$resAlias.${ER.ID} ${ColumnAliases.EnvReservations.ID}
           | ,$resAlias.${ER.START_DATE} ${ColumnAliases.EnvReservations.START_DATE}
           | ,$resAlias.${ER.END_DATE} ${ColumnAliases.EnvReservations.END_DATE}
           | ,$resAlias.${ER.NOTE} ${ColumnAliases.EnvReservations.NOTE}
           | ,$resAlias.${ER.ENVIRONMENT_UID} ${ColumnAliases.EnvReservations.ENVIRONMENT_UID}
           | FROM ${ER.TABLE} $resAlias
           | LEFT JOIN ${ENV_RES_TO_APP.TABLE} $resToAppAlias
           |  ON $resAlias.${ER.CI_UID} = $resToAppAlias.${ENV_RES_TO_APP.RESERVATION_UID}
           | LEFT JOIN ${APPLICATIONS.TABLE} $appAlias
           |  ON $resToAppAlias.${ENV_RES_TO_APP.APPLICATION_UID} = $appAlias.${APPLICATIONS.CI_UID}
        """.stripMargin
      super.select(stmt)

      super.build()
    }

    def withApplicationTitles(titles: Iterable[String]): BaseReservationSqlBuilder = {
      conditions ++= whereInCondition(s"$appAlias.${APPLICATIONS.TITLE}", titles).toList
      this
    }

    def from(date: Date): BaseReservationSqlBuilder = {
      conditions += Sql(s"$resAlias.${ER.END_DATE} >= ?", Seq(date))
      this
    }

    def to(date: Date): BaseReservationSqlBuilder = {
      conditions += Sql(s"$resAlias.${ER.START_DATE} < ?", Seq(date))
      this
    }

    override def newInstance: BaseReservationSqlBuilder = new BaseReservationSqlBuilder(aliasPrefix)

  }

}
