package com.xebialabs.xlrelease.api.v1.impl;

import java.util.List;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.api.v1.EnvironmentStageApi;
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentStageFilters;
import com.xebialabs.xlrelease.api.v1.form.EnvironmentStageForm;
import com.xebialabs.xlrelease.api.v1.views.EnvironmentStageView;
import com.xebialabs.xlrelease.domain.environments.EnvironmentStage;
import com.xebialabs.xlrelease.environments.service.EnvironmentSecurity;
import com.xebialabs.xlrelease.environments.service.EnvironmentStageService;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.xlrelease.api.v1.views.EnvironmentStageView.fromEnvironmentStage;

@Controller
public class EnvironmentStageApiImpl implements EnvironmentStageApi {

    private EnvironmentStageService environmentStageService;
    private EnvironmentSecurity environmentSecurity;

    @Autowired
    public EnvironmentStageApiImpl(EnvironmentStageService environmentStageService, EnvironmentSecurity environmentSecurity) {
        this.environmentStageService = environmentStageService;
        this.environmentSecurity = environmentSecurity;
    }

    @Timed
    @Override
    public EnvironmentStageView createStage(EnvironmentStageForm stageForm) {
        environmentSecurity.checkCreate();
        return fromEnvironmentStage(
                environmentStageService.createStage(stageForm.toStage())
        );
    }

    @Timed
    @Override
    public EnvironmentStageView getStageById(String environmentStageId) {
        environmentSecurity.checkView();
        return fromEnvironmentStage(this.environmentStageService.findStageById(environmentStageId));
    }

    @Timed
    @Override
    public EnvironmentStageView updateStage(String environmentStageId, EnvironmentStageForm stageForm) {
        environmentSecurity.checkEdit();
        EnvironmentStage stage = stageForm.toStage();
        stage.setId(environmentStageId);
        return fromEnvironmentStage(environmentStageService.updateStage(stage));
    }

    @Timed
    @Override
    public List<EnvironmentStageView> searchStages(EnvironmentStageFilters filters) {
        environmentSecurity.checkView();
        return environmentStageService.searchStages(filters)
                .stream().map(EnvironmentStageView::fromEnvironmentStage).collect(Collectors.toList());
    }

    @Timed
    @Override
    public void delete(String environmentStageId) {
        environmentSecurity.checkEdit();
        environmentStageService.deleteStage(environmentStageId);
    }

    @Timed
    @Override
    public EnvironmentStage create(EnvironmentStage environmentStage) {
        environmentSecurity.checkCreate();
        return environmentStageService.createStage(environmentStage);
    }

    @Timed
    @Override
    public EnvironmentStage getById(String environmentStageId) {
        environmentSecurity.checkView();
        return this.environmentStageService.findStageById(environmentStageId);
    }

    @Timed
    @Override
    public EnvironmentStage update(EnvironmentStage environmentCategory) {
        environmentSecurity.checkEdit();
        return environmentStageService.updateStage(environmentCategory);
    }

    @Timed
    @Override
    public List<EnvironmentStage> search(EnvironmentStageFilters filters) {
        environmentSecurity.checkView();
        return environmentStageService.searchStages(filters);
    }
}
