import React, { useEffect, useState } from 'react';
import sortBy from 'lodash/sortBy';
import clone from 'lodash/clone';
import differenceWith from 'lodash/differenceWith';
import isEmpty from 'lodash/isEmpty';
import isEqual from 'lodash/isEqual';
import reject from 'lodash/reject';
import some from 'lodash/some';

import { SidebarLabelList } from './sidebar-label-list';
import WidgetLinkedLabels from './../../shared/widgets/widget-linked-labels';

import './../../../styles/common.less';
import './environment-form-component.less';
import { itemAlreadyDefined } from '@xlr-ui/app/react/utils/validators';
import { Environment, Label, Stage } from '../../../application-list.types';
import { CssGrid, DotInputSelect, DotInputText, DotThemeProvider } from '@digital-ai/dot-components';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { getFormEnvironmentModel, getEnvironments, getLabelsList, getStageList } from '../ducks/environment-list.selectors';
import { environmentForm } from '../ducks/environment-form.reducer';
import { labelsState } from '../ducks/labels.reducer';
import { GridCell } from '@xlr-ui/app/features/tasks/components/rails/grid.layout';

const nameFormMetadata = {
    label: 'Name',
    placeholder: 'Add...',
    description: 'Name of the environment',
    required: true,
};
const descriptionFormMetadata = {
    label: 'Description',
    placeholder: 'Set description...',
    description: 'Description of the environment',
};

const selectFormMetadata = {
    description: 'Name of the stage',
    label: 'Stage',
    placeholder: 'Select stage...',
    required: true,
};

const findEnvironmentById = (environments: Array<Environment>, id: string) => {
    return environments.find((e) => e.id === id) || {};
};

const { searchLabels, saveLabel } = labelsState.actions;
const { initForm, setEnvironmentModel, pristineFormChanged, setSaveEnabled } = environmentForm.actions;

export interface EnvironmentFormProps {
    environmentId: string;
    folderId: string;
}

export const EnvironmentFormComponent = ({ environmentId, folderId }: EnvironmentFormProps) => {
    const dispatch = useAppDispatch();
    const [isNameValid, setNameValid] = useState(true);
    const nameErrorMsg = 'Environment already exists';
    const defaultOption = { option: selectFormMetadata.placeholder };
    const [selectedStage, setSelectedStage] = useState(defaultOption.option);
    useEffect(() => {
        dispatch(initForm({ envId: environmentId, folderId }));
        dispatch(searchLabels(''));
        dispatch(setSaveEnabled(false));
    }, []);

    const stages = useAppSelector(getStageList);
    const labelList = useAppSelector(getLabelsList);
    const environmentModel = useAppSelector(getFormEnvironmentModel);
    const environments = useAppSelector(getEnvironments);

    useEffect(() => {
        if (!isEmpty(environmentModel.stage.title)) {
            setSelectedStage(environmentModel.stage.title);
        }
    }, []);

    useEffect(() => {
        if (!isEmpty(environmentModel.stage.title)) {
            setSelectedStage(environmentModel.stage.title);
        } else {
            setSelectedStage(defaultOption.option);
        }
    }, [environmentModel.stage.title]);

    const onPristinityChange = (pristine: boolean) => {
        if (!pristine) {
            dispatch(pristineFormChanged(pristine));
        }
    };

    const onNameChange = (title: string) => {
        if (title !== environmentModel.title) {
            dispatch(
                setEnvironmentModel({
                    ...environmentModel,
                    title,
                }),
            );
        }
        const error = itemAlreadyDefined(environments, environmentModel.id, 'title', title);

        setNameValid(!error);
        dispatch(setSaveEnabled(isNameValid && !isEmpty(title) && selectedStage !== defaultOption.option));
    };

    const onStageChange = (stageTitle: string) => {
        if (stageTitle && stageTitle !== environmentModel.stage.title) {
            setSelectedStage(stageTitle);
            dispatch(
                setEnvironmentModel({
                    ...environmentModel,
                    stage: stages.find((stage: Stage) => stage.title === stageTitle) || environmentModel.stage,
                }),
            );
            dispatch(setSaveEnabled(isNameValid && !isEmpty(environmentModel.title) && stageTitle !== defaultOption.option));
        }
    };

    const onDescriptionChange = (description: string) => {
        if (description !== environmentModel.description) {
            dispatch(
                setEnvironmentModel({
                    ...environmentModel,
                    description,
                }),
            );
            dispatch(setSaveEnabled(isNameValid && !isEmpty(environmentModel.title) && selectedStage !== defaultOption.option));
        }
    };

    const onSaveLabel = (label: Label) => {
        dispatch(saveLabel(label));
        dispatch(searchLabels(''));
    };

    const onAddLabel = (envLabel: Label) => {
        const labels = clone(environmentModel.labels);

        if (!some(environmentModel.labels, { id: envLabel.id })) {
            labels.push(envLabel);
        }

        dispatch(
            setEnvironmentModel({
                ...environmentModel,
                labels,
            }),
        );
        dispatch(setSaveEnabled(isNameValid && !isEmpty(environmentModel.title) && selectedStage !== defaultOption.option));
        onPristinityChange(false);
    };

    const onRemoveLabel = (envLabel: Label) => {
        const labels = reject(environmentModel.labels, { id: envLabel.id });
        dispatch(
            setEnvironmentModel({
                ...environmentModel,
                labels,
            }),
        );
        dispatch(setSaveEnabled(isNameValid));
        onPristinityChange(false);
    };

    const renderSidebarContent = () => {
        const filteredLabelList = differenceWith(labelList, environmentModel.labels, isEqual);

        return <SidebarLabelList labelList={filteredLabelList} onClickLabel={onAddLabel} saveLabel={onSaveLabel} />;
    };

    const env = findEnvironmentById(environments, environmentModel.id) as Environment;
    const subtitle = isEmpty(env) ? 'New Environment' : env.title;

    const stageOptions = sortBy(
        stages.map((stage) => {
            return { option: stage.title };
        }),
        ['option'],
    );

    return (
        <DotThemeProvider>
            <div className="environment-form-container">
                <CssGrid>
                    <GridCell span={9} start={1}>
                        <div className="environment-form-content">
                            <h4 className="page-title">Environments / {subtitle}</h4>
                            <div className="environment-form">
                                <DotInputText
                                    autoFocus={true}
                                    className={'dot-input-name'}
                                    error={!isNameValid}
                                    helperText={isNameValid ? '' : nameErrorMsg}
                                    id={'environment-form-title'}
                                    label={nameFormMetadata.label}
                                    maxLength={255}
                                    name={'environment-form-title'}
                                    onChange={(e) => onNameChange(e.target.value)}
                                    persistentLabel={true}
                                    placeholder={nameFormMetadata.placeholder}
                                    required={nameFormMetadata.required}
                                    value={environmentModel.title}
                                />
                                <DotInputSelect
                                    className={'select'}
                                    id={'envStageSelector'}
                                    label={selectFormMetadata.label}
                                    name={'envStageSelector'}
                                    onChange={(e) => onStageChange(e.target.value)}
                                    options={[defaultOption, ...stageOptions]}
                                    persistentLabel={true}
                                    required={selectFormMetadata.required}
                                    value={selectedStage}
                                />
                                <DotInputText
                                    className={'dot-input-desc'}
                                    id={'env-desc-input'}
                                    label={descriptionFormMetadata.label}
                                    maxLength={1024}
                                    multiline={true}
                                    name={'env-desc-input'}
                                    onChange={(e) => onDescriptionChange(e.target.value)}
                                    persistentLabel={true}
                                    placeholder={descriptionFormMetadata.placeholder}
                                    size={'medium'}
                                    value={environmentModel.description}
                                />
                                <WidgetLinkedLabels labelList={environmentModel.labels} onDelete={onRemoveLabel} />
                            </div>
                        </div>
                    </GridCell>
                    <GridCell span={3} start={11}>
                        <div className="environment-form-sidebar">{renderSidebarContent()}</div>
                    </GridCell>
                </CssGrid>
            </div>
        </DotThemeProvider>
    );
};
