import React from 'react';

import { StageListComponent } from './stage-list-component';
import { StageListHeaderComponent } from './stage-list-header-component';
import { StageListItemComponent } from './stage-list-item-component';
import { NoData } from '@xlr-ui/app/react/components';
import { ReactWrapper, preparePortalContainer, mountComponentWithStore } from '@xlr-ui/tests/unit/testing-utils';
import { stageState } from '../ducks/stage.reducer';
import { DotConfirmationDialog, DotInputText } from '@digital-ai/dot-components';
import { StageEditModal } from './stage-edit-modal';

const { searchStages, deleteStage, saveStage } = stageState.actions;

describe('Stage list component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const emptyState = {
        stageState: {
            stageList: [],
        },
    };
    const state = {
        stageState: {
            stageList: [
                {
                    id: 'Stage1',
                    title: 'DEV',
                },
                {
                    id: 'Stage2',
                    title: 'TEST',
                },
            ],
        },
    };

    const mount = () => {
        wrapper = mountComponentWithStore(<StageListComponent />, dispatch, state);
    };

    beforeEach(() => {
        preparePortalContainer();
        mount();
    });

    it('should render with model', () => {
        const header = wrapper.find(StageListHeaderComponent);
        expect(header.exists()).toBeTruthy();
        expect(wrapper.find('h4').text()).toBe('Stages (2)');
        expect(wrapper.find(NoData).length).toBe(0);

        const items = wrapper.find(StageListItemComponent);
        expect(items).toHaveLength(2);

        //delete, create and update modal
        const modals = wrapper.find(DotConfirmationDialog);
        expect(modals).toHaveLength(2);
    });

    it('should filter stages list', () => {
        wrapper.find(StageListHeaderComponent).invoke('onFilterChange')('dEv');
        expect(dispatch).toBeCalledWith(searchStages('dEv'));
    });

    it('should render delete modal', () => {
        const item = wrapper.find(StageListItemComponent).at(0);
        item.invoke('onDelete')(state.stageState.stageList[0]);
        // show delete modal
        const deleteModal = wrapper.find(DotConfirmationDialog).at(0);
        expect(deleteModal.prop('title')).toBe('Delete stage');
    });

    it('should render update modal', () => {
        // show update modal
        const updateModal = wrapper.find(DotConfirmationDialog).at(1);
        expect(updateModal.prop('open')).toBe(false);
        const item = wrapper.find(StageListItemComponent).at(0);
        item.invoke('onEdit')(state.stageState.stageList[0]);
        expect(wrapper.find(StageEditModal).find(DotInputText).prop('defaultValue')).toBe(state.stageState.stageList[0].title);
    });

    it('should delete stage on closing modal', () => {
        const item = wrapper.find(StageListItemComponent).at(1);
        item.invoke('onDelete')(state.stageState.stageList[1]);
        const deleteModal = wrapper.find(DotConfirmationDialog).at(0);
        deleteModal.invoke('onSubmit')();
        expect(dispatch).toHaveBeenCalledWith(deleteStage(state.stageState.stageList[1]));
    });

    it('should update stage title', () => {
        const updateModal = wrapper.find(StageEditModal);
        const item = wrapper.find(StageListItemComponent).at(0);
        item.invoke('onEdit')(state.stageState.stageList[0]);
        updateModal.update();
        updateModal.invoke('onSave')({ ...state.stageState.stageList[0], title: 'changed' });
        expect(dispatch).toBeCalledWith(saveStage({ ...state.stageState.stageList[0], title: 'changed' }));
    });

    it('should render NoData placeholder for empty list', () => {
        const stageListComponent = mountComponentWithStore(<StageListComponent />, dispatch, emptyState);
        expect(stageListComponent.find('h4').text()).toBe('Stages (0)');
        expect(stageListComponent.find(NoData).length).toBe(1);
    });

    it('should cancel edit modal', () => {
        const component = mountComponentWithStore(<StageListComponent />, dispatch, emptyState);
        const editModal = component.find(StageEditModal);
        expect(editModal).toExist();
        editModal.invoke('onCancel')();
        expect(editModal.prop('isOpened')).toBeFalsy();
    });
});
