package com.xebialabs.xlrelease.api.v1.impl;

import java.util.List;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.api.v1.EnvironmentApi;
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentFilters;
import com.xebialabs.xlrelease.api.v1.form.EnvironmentForm;
import com.xebialabs.xlrelease.api.v1.views.BaseApplicationView;
import com.xebialabs.xlrelease.api.v1.views.EnvironmentReservationView;
import com.xebialabs.xlrelease.api.v1.views.EnvironmentView;
import com.xebialabs.xlrelease.domain.environments.Application;
import com.xebialabs.xlrelease.domain.environments.Environment;
import com.xebialabs.xlrelease.domain.environments.EnvironmentReservation;
import com.xebialabs.xlrelease.environments.service.*;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.xlrelease.api.v1.views.EnvironmentView.fromEnvironment;

@Controller
public class EnvironmentApiImpl implements EnvironmentApi {

    private EnvironmentService environmentService;
    private EnvironmentSecurity environmentSecurity;
    private ApplicationService applicationService;
    private EnvironmentReservationService environmentReservationService;

    @Autowired
    public EnvironmentApiImpl(EnvironmentService environmentService, EnvironmentSecurity environmentSecurity, ApplicationService applicationService, EnvironmentReservationService environmentReservationService) {
        this.environmentService = environmentService;
        this.environmentSecurity = environmentSecurity;
        this.applicationService = applicationService;
        this.environmentReservationService = environmentReservationService;
    }

    @Timed
    @Override
    public EnvironmentView createEnvironment(EnvironmentForm environmentForm) {
        environmentSecurity.checkCreate();
        return fromEnvironment(environmentService.createEnvironment(environmentForm.toEnvironment()));
    }

    @Timed
    @Override
    public EnvironmentView getEnvironment(String environmentId) {
        environmentSecurity.checkView();
        return fromEnvironment(environmentService.findEnvironmentById(environmentId));
    }

    @Timed
    @Override
    public EnvironmentView updateEnvironment(String environmentId, EnvironmentForm environmentForm) {
        environmentSecurity.checkEdit();
        Environment environment = environmentForm.toEnvironment();
        environment.setId(environmentId);
        return fromEnvironment(environmentService.updateEnvironment(environment));
    }

    @Timed
    @Override
    public List<EnvironmentView> searchEnvironments(EnvironmentFilters environmentFilters) {
        environmentSecurity.checkView();
        return environmentService.searchEnvironments(environmentFilters)
                .stream().map(EnvironmentView::fromEnvironment).collect(Collectors.toList());
    }

    @Timed
    @Override
    public List<EnvironmentReservationView> getReservationsForEnvironment(String environmentId) {
        environmentSecurity.checkView();
        return environmentReservationService.getReservationsByEnvironmentId(environmentId)
                .stream().map(EnvironmentReservationView::fromEnvironmentReservation).collect(Collectors.toList());
    }

    @Timed
    @Override
    public List<BaseApplicationView> getDeployableApplicationsForEnvironment(String environmentId) {
        environmentSecurity.checkView();
        return applicationService.getApplicationsDeployableOnEnvironment(environmentId)
                .stream().map(BaseApplicationView::fromApplication).collect(Collectors.toList());
    }

    @Timed
    @Override
    public void delete(String environmentId) {
        environmentSecurity.checkEdit();
        var connectedApps = applicationService.findByEnvironment(environmentId);
        if (connectedApps.isEmpty()) {
            environmentService.deleteEnvironment(environmentId);
        } else {
            throw new IllegalArgumentException(String.format("Environment is connected to applications: %s", connectedApps.mkString(", ")));
        }
    }

    @Timed
    @Override
    public Environment create(Environment environment) {
        environmentSecurity.checkCreate();
        return environmentService.createEnvironment(environment);
    }

    @Timed
    @Override
    public Environment getById(String environmentId) {
        environmentSecurity.checkView();
        return environmentService.findEnvironmentById(environmentId);
    }

    @Timed
    @Override
    public Environment update(Environment environment) {
        environmentSecurity.checkEdit();
        return environmentService.updateEnvironment(environment);
    }

    @Timed
    @Override
    public List<Environment> search(EnvironmentFilters filters) {
        environmentSecurity.checkView();
        return environmentService.searchEnvironments(filters);
    }

    @Timed
    @Override
    public List<EnvironmentReservation> getReservations(String environmentId) {
        environmentSecurity.checkView();
        return environmentReservationService.getReservationsByEnvironmentId(environmentId);
    }

    @Timed
    @Override
    public List<Application> getDeployableApplications(String environmentId) {
        environmentSecurity.checkView();
        return applicationService.getApplicationsDeployableOnEnvironment(environmentId);
    }
}
