import React, { useEffect, useState } from 'react';

import { EnvironmentListItemComponent } from './environment-list-item-component';
import { AlertMessage, alertType, HelpBlock, NoData } from '@xlr-ui/app/react/components';

import './../../../styles/common.less';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { getEnvironmentListState } from '../ducks/environment-list.selectors';
import { Environment } from '../../../application-list.types';
import { EnvironmentListHeaderComponent } from './environment-list-header-component';
import { environmentLists } from '../ducks/environment-list.reducer';
import { DotConfirmationDialog, DotIcon, DotThemeProvider, DotTypography } from '@digital-ai/dot-components';

const helpMessage = `Use Environments to define environments that are available in your organization. You can also use Environments to filter the environment scheduler and dashboards.`;
const { initEnvironmentList, deleteEnvironment } = environmentLists.actions;

export interface EnvironmentListProps {
    folderId: string;
}

export const EnvironmentListComponent = ({ folderId }: EnvironmentListProps) => {
    const dispatch = useAppDispatch();
    const [show, setShow] = useState(false);
    const [environmentForDelete, setEnvironmentForDelete] = useState(undefined as unknown as Environment);

    useEffect(() => {
        dispatch(initEnvironmentList(folderId));
    }, []);

    const environments = useAppSelector(getEnvironmentListState).environments;

    const onCloseDeleteModal = (action: string) => {
        if (action === 'delete') {
            dispatch(deleteEnvironment(environmentForDelete));
        }
        setShow(false);
        setEnvironmentForDelete(undefined as unknown as Environment);
    };

    const openConfirmationDialog = (environment: Environment) => {
        setShow(true);
        setEnvironmentForDelete(environment);
    };

    const renderHeader = () => (
        <div className="application-list-header">
            <span className="col-xs-7">Name</span>
            <span className="col-xs-2">Stage</span>
            <div className="col-xs-3">
                <span className="pull-right action-section">Actions</span>
            </div>
        </div>
    );

    const renderEnvironmentList = () => {
        return (
            <React.Fragment>
                {renderHeader}
                {environments.map((environment) => (
                    <EnvironmentListItemComponent
                        environment={environment}
                        folderId={environment.folderId}
                        key={environment.title}
                        onDelete={openConfirmationDialog}
                    />
                ))}
            </React.Fragment>
        );
    };

    return (
        <DotThemeProvider>
            <EnvironmentListHeaderComponent folderId={folderId} />
            <div className="header-shadow"></div>
            <div className="margin environment-panel">
                <h4 className="page-title">Environments ({environments.length})</h4>
                <HelpBlock className="help-message-container">{helpMessage}</HelpBlock>
                {environments.length ? renderEnvironmentList() : <NoData message="No environments found." />}
            </div>
            <DotConfirmationDialog
                className="xl-default-modal"
                message={
                    <div>
                        <AlertMessage alertType={alertType.WARNING}>
                            <p>Deleting an environment can lead to missing data for connected resources on reporting and environment scheduling.</p>
                        </AlertMessage>
                        <p>
                            You are about to delete environment <strong>{environmentForDelete ? environmentForDelete.title : ''}</strong>.
                        </p>
                    </div>
                }
                onCancel={() => onCloseDeleteModal('cancel')}
                onSubmit={() => onCloseDeleteModal('delete')}
                open={show}
                submitButtonProps={{ label: 'Delete Environment', startIcon: <DotIcon iconId="delete" />, type: 'destructive' }}
                title={<DotTypography variant="h1">Delete environment</DotTypography>}
            />
        </DotThemeProvider>
    );
};
